#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <fcntl.h>
#include <stdlib.h>
#include <malloc.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include "slacker.h"

char *ourname;
int blocksize = 0;

static void usage(void)
{
	fprintf(stderr,
		"Usage: %s -n nreads [-klop] [-a align] [-B blocksize] [-c chunksize(bytes)] [-S usecs] filename\n"
	"          -a align      : O_DIRECT memory alignment (bytes)\n"
	"          -k            : -c is in kbytes\n"
	"          -l            : read latency histogram\n"
	"          -o            : use O_DIRECT\n"
	"          -p            : show progress dots\n"
	"          -S            : sleep this many usecs between read()s\n",
		ourname);
	exit(1);
}

static void set_blocksize(int fd)
{
	if (blocksize) {
		if (ioctl(fd, BLKBSZSET, &blocksize) < 0) {
			perror("ioctl");
			exit(1);
		}
	}
}

int main(int argc, char *argv[])
{
	char *filename;
	unsigned long nreads = 0;
	int fd;
	int chunk = 1024 * 1024;
	char *buffer;
	int c;
	int o_direct = 0;
	int kilobytes = 0;
	int latency = 0;
	int progress = 0;
	int align = 4096;
	unsigned *histogram = NULL;
	unsigned histogram_size = 0;
	unsigned sleep_usecs = 0;

	ourname = argv[0];

	if (argc < 2)
		usage();

	while ((c = getopt(argc, argv, "klopa:B:n:c:S:")) != -1) {
		switch (c) {
		case 'a':
			align = strtol(optarg, NULL, 10);
			break;
		case 'B':
			blocksize = strtol(optarg, NULL, 10);
			break;
		case 'c':
			chunk = strtol(optarg, NULL, 10);
			break;
		case 'k':
			kilobytes++;
			break;
		case 'l':
			latency++;
			break;
		case 'o':
			o_direct = O_DIRECT;
			break;
		case 'p':
			progress = 1;
			break;
		case 'n':
			nreads = strtol(optarg, NULL, 10);
			break;
		case 'S':
			sleep_usecs = strtol(optarg, NULL, 10);
			break;
		default:
			usage();
		}
	}

	if (kilobytes)
		chunk *= 1024;

	if (nreads == 0)
		usage();
	if (optind == argc)
		usage();
	filename = argv[optind++];
	if (optind != argc)
		usage();

	buffer = memalign(align, chunk);

	fd = open(filename, O_RDONLY|o_direct);
	if (fd < 0) {
		fprintf(stderr, "%s: cannot open %s: %s\n",
			argv[0], filename, strerror(errno));
		exit(1);
	}
	set_blocksize(fd);

	while (nreads--) {
		struct timeval start;
		struct timeval stop;

		if (latency)
			gettimeofday(&start, NULL);
		if (read(fd, buffer, chunk) != chunk) {
			perror("write");
			exit(1);
		}
		if (latency) {
			unsigned long long delta_ms;

			gettimeofday(&stop, NULL);
			delta_ms = (unsigned long long)stop.tv_sec * 1000 +
					stop.tv_usec / 1000;
			delta_ms -= (unsigned long long)start.tv_sec * 1000 +
					start.tv_usec / 1000;
			if (delta_ms + 1 > histogram_size) {
				histogram = realloc(histogram, (delta_ms + 1) *
						sizeof(*histogram));
				memset(histogram + histogram_size, 0,
					((delta_ms + 1) - histogram_size) *
						sizeof(*histogram));
				histogram_size = delta_ms + 1;
			}
			histogram[delta_ms]++;
		}
		if (sleep_usecs)
			usleep(sleep_usecs);
		if (progress) {
			putchar('.');
			fflush(stdout);
		}
	}
	if (progress)
		putchar('\n');
	close(fd);
	if (latency) {
		unsigned i;

		for (i = 0; i < histogram_size; i++) {
			if (!histogram[i])
				continue;
			printf("%3d.%03d: %u\n", i / 1000, i % 1000,
				histogram[i]);
		}
	}
	exit(0);
}
