/*
 * Read a file, via mmap
 */

#undef _XOPEN_SOURCE	/* madvise.  grr. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include "slacker.h"

char *ourname;
int do_madv_sequential;
int do_madv_random;
int do_mmap;
int hop = 4096;
int read_buf_size = 8192;
int o_direct = 0;
long long nbytes = 0x7fffffffffffffffLL;
unsigned align_offset = 0;
int blocksize = 0;

ssize_t pread(int fd, void *buf, size_t count, off_t offset);

void usage(void)
{
	fprintf(stderr,
		"Usage: %s [-kmors] [-a N] [-h N] [-b N] [-B N] [-n N] filename\n"
		"         -a N : O_DIRECT memory alignment (bytes)\n"
		"         -b N : read buffer size (bytes)\n"
		"         -B N : blocksize\n"
		"         -h N : Hop-factor (bytes)\n"
		"         -k   : -b and -h are in kilobytes\n"
		"         -m   : Use mmap(), not read\n"
		"         -n N : Number of megabytes to read\n"
		"         -o   : Use O_DIRECT\n"
		"         -r   : Use madvise(MADV_RANDOM)\n"
		"         -s   : Use madvise(MADV_SEQUENTIAL)\n",
				ourname);
	exit(1);
}

static void *round_up(void *ptr, unsigned align, unsigned offset)
{
	unsigned long ret = (unsigned long)ptr;

	ret = ((ret + align - 1) & ~(align - 1));
	ret += offset;
	return (void *)ret;
}

static void set_blocksize(int fd)
{
	if (blocksize) {
		if (ioctl(fd, BLKBSZSET, &blocksize) < 0) {
			perror("BLKBSZSET");
			exit(1);
		}
	}
}

int doit_mmap(char *filename)
{
	int fd;
	char *map;
	int i;
	struct stat64 statbuf;
	int ret = 0;
	long long bytes_to_read;

	fd = open(filename, O_RDONLY|o_direct);
	if (fd < 0) {
		fprintf(stderr, "%s: Cannot open `%s': %s\n",
			ourname, filename, strerror(errno));
		exit(1);
	}

	set_blocksize(fd);
	fstat64(fd, &statbuf);
	map = mmap(0, statbuf.st_size, PROT_READ, MAP_PRIVATE, fd, 0);
	if (map == MAP_FAILED) {
		fprintf(stderr, "%s: mmap failed: %s\n",
			ourname, strerror(errno));
		exit(1);
	}

	if (do_madv_sequential) {
		if (madvise(map, statbuf.st_size, MADV_SEQUENTIAL) < 0) {
			perror("madvise");
			exit(1);
		}
	}
	if (do_madv_random) {
		if (madvise(map, statbuf.st_size, MADV_RANDOM) < 0) {
			perror("madvise");
			exit(1);
		}
	}
	bytes_to_read = statbuf.st_size;
	if (bytes_to_read > nbytes)
		bytes_to_read = nbytes;
	for (i = 0; i < statbuf.st_size; i += hop) {
		ret += map[i];
		bytes_to_read -= hop;
		if (bytes_to_read <= 0)
			break;
	}
	return ret;
}

void doit_read(char *filename)
{
	int fd;
	loff_t off = 0;
	char *_buf;
	char *buf;
	struct stat64 statbuf;
	long long bytes_to_read;
	int ret;

	fd = open(filename, O_RDONLY|o_direct);
	if (fd < 0) {
		fprintf(stderr, "%s: Cannot open `%s': %s\n",
			ourname, filename, strerror(errno));
		exit(1);
	}

	_buf = malloc(read_buf_size + 40960);
	buf = round_up(_buf, 4096, align_offset);
	if (align_offset)
		printf("IO buffer is at %p\n", buf);
	fstat64(fd, &statbuf);
	set_blocksize(fd);
	bytes_to_read = statbuf.st_size;
	if (bytes_to_read > nbytes || bytes_to_read == 0)
		bytes_to_read = nbytes;
	while ((ret = pread(fd, buf, read_buf_size, off)) == read_buf_size) {
		off += hop;
		bytes_to_read -= hop;
		if (bytes_to_read <= 0)
			break;
	}
	if (ret < 0) {
		perror("pread");
		exit(1);
	}
	free(_buf);
}

int main(int argc, char *argv[])
{
	char *filename;
	int c;
	int kbytes = 1;

	ourname = argv[0];
	while ((c = getopt(argc, argv, "kmorsa:b:B:h:n:")) != -1) {
		switch (c) {
		case 'a':
			align_offset = strtol(optarg, NULL, 10);
			break;
		case 'b':
			read_buf_size = strtol(optarg, NULL, 10);
			break;
		case 'B':
			blocksize = strtol(optarg, NULL, 10);
			break;
		case 'h':
			hop = strtol(optarg, NULL, 10);
			break;
		case 'k':
			kbytes = 1024;
			break;
		case 'm':
			do_mmap++;
			break;
		case 'n':
			nbytes = strtol(optarg, NULL, 10);
			nbytes *= 1024LL * 1024LL;
			break;
		case 'o':
			o_direct = O_DIRECT;
			break;
		case 'r':
			do_madv_random++;
			break;
		case 's':
			do_madv_sequential++;
			break;
		default:
			usage();
		}
	}

	read_buf_size *= kbytes;
	hop *= kbytes;

	if (optind == argc)
		usage();
	filename = argv[optind++];
	if (do_mmap)
		doit_mmap(filename);
	else
		doit_read(filename);
	exit(0);
}
