/**********************************************************************
   Copyright (C) Christopher Yeoh <cyeoh@samba.org> 2005
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**********************************************************************/
#include "RemoteConnection.H"

#include <assert.h>
#include <errno.h>

RemoteConnection::RemoteConnection(int ConnectionFD)
    : connectionFD(ConnectionFD), 
      ringBufferReadStart(0), ringBufferReadEnd(0),
      ringBufferWriteStart(0), ringBufferWriteEnd(0),
      msgStart(0)
{
}

int
RemoteConnection::getFD() const
{
    return connectionFD;
}

bool
RemoteConnection::pendingDataToWrite() const
{
    return ringBufferWriteStart != ringBufferWriteEnd;
}

bool
RemoteConnection::haveMessageToRead() const
{
    return msgsToRead.size()!=0;
}

Message *
RemoteConnection::getNextMessage()
{
    assert(haveMessageToRead());

    MsgMarker msg = msgsToRead.front();
    msgsToRead.pop();

    assert(msg.length != 0);

    char *messageBuffer = (char *)malloc(msg.length);
    if (messageBuffer==NULL) {
	printf("failed to malloc %i bytes\n", msg.length);
	assert(0);
    }
    
    // Copy data to message buffer
    if (msg.start + msg.length >= REMOTE_RING_BUFFER_SIZE) {
	// read across end of ring buffer
	const int bytesToEnd = REMOTE_RING_BUFFER_SIZE - msg.start;
	memcpy(messageBuffer, ringBufferRead+msg.start, bytesToEnd);
	memcpy(messageBuffer+bytesToEnd, ringBufferRead, 
	       msg.length - bytesToEnd);
    } else {
	memcpy(messageBuffer, ringBufferRead+msg.start, msg.length);
    }

    // free up portion of read ring buffer no longer used
    ringBufferReadStart = msg.start + msg.length;
    if (ringBufferReadStart>REMOTE_RING_BUFFER_SIZE)
	ringBufferReadStart =- REMOTE_RING_BUFFER_SIZE;

    return (Message *)messageBuffer;
}

void
RemoteConnection::writeMessage(Message *MsgToSend)
{
    const char *messageBuffer = (const char *)MsgToSend;
    int messageBufferSize = MsgToSend->length;

    int bufferLeft;
    if (ringBufferWriteStart < ringBufferWriteEnd) {
	bufferLeft = REMOTE_RING_BUFFER_SIZE - ringBufferWriteEnd
	    + ringBufferWriteStart;
    } else if (ringBufferWriteStart==ringBufferWriteEnd) {
	bufferLeft = REMOTE_RING_BUFFER_SIZE;
    } else {
	bufferLeft = ringBufferWriteStart - ringBufferWriteEnd;
    }

    // Check we have enough space left in the ring buffer
    // Should really handle it better than this - disconnect dead client?
    assert(bufferLeft>messageBufferSize);

    if (ringBufferWriteEnd+messageBufferSize<REMOTE_RING_BUFFER_SIZE) {
	memcpy(ringBufferWrite+ringBufferWriteEnd, 
	       messageBuffer, messageBufferSize);
	ringBufferWriteEnd += messageBufferSize;
    } else {
	const int bytesToEnd = REMOTE_RING_BUFFER_SIZE - ringBufferWriteEnd;
	memcpy(ringBufferWrite+ringBufferWriteEnd, messageBuffer, bytesToEnd);
	memcpy(ringBufferWrite, messageBuffer+bytesToEnd, 
	       messageBufferSize - bytesToEnd);
	ringBufferWriteEnd = messageBufferSize - bytesToEnd;
    }

    // Attempt to write as much data as possible
    writeData();
}

void
RemoteConnection::readData() 
{
    int bytesRead;
    int bytesToEnd;

    if (ringBufferReadStart <= ringBufferReadEnd) {
	bytesToEnd = REMOTE_RING_BUFFER_SIZE - ringBufferReadEnd;
    } else {
	bytesToEnd = ringBufferReadStart - ringBufferReadEnd;
    }

    bytesRead = read(connectionFD, ringBufferRead+ringBufferReadEnd,
		     bytesToEnd);

    if (bytesRead<0) {
	perror("RemoteConnection::readData read failed");
	// Need to handle disconnects from client
	// raise an exception?
	assert(0);
    } else if (bytesRead==0) {
	// This should not have happened
 	assert(0);
    }

    ringBufferReadEnd+=bytesRead;
    // Wrap around ring buffer
    if (ringBufferReadEnd>=REMOTE_RING_BUFFER_SIZE) {
	ringBufferReadEnd -= REMOTE_RING_BUFFER_SIZE;
    }

    // Check for new message

    // Work out how many bytes we have left to read
    unsigned int bytesAvailableToRead;
    if (ringBufferReadStart<ringBufferReadEnd) {
	bytesAvailableToRead = ringBufferReadEnd - ringBufferReadStart;
    } else {
	bytesAvailableToRead = REMOTE_RING_BUFFER_SIZE - ringBufferReadStart
	    + ringBufferReadEnd;
    }

    // Assume all messages are of struct Message format
    // so we have header followed by length of entire message
    // could get multiple messages in single read
    int msgStartPoint = ringBufferReadStart;
    while (bytesAvailableToRead>=sizeof(Message)) {
	Message *msg = (Message *)(ringBufferRead+msgStartPoint);
	assert(msg->length>=sizeof(Message));
// 	printf("Looking for message %i bytes long (have %i)\n", 
// 	       msg->length, bytesAvailableToRead);
	if (bytesAvailableToRead>=msg->length) {
	    // We have the entire message
	    MsgMarker marker;
	    marker.start = msgStartPoint;
	    marker.length = msg->length;
	    msgsToRead.push(marker);
	    
	    bytesAvailableToRead -= msg->length;
	    msgStartPoint += msg->length;
	    if (msgStartPoint>REMOTE_RING_BUFFER_SIZE) {
		msgStartPoint -= REMOTE_RING_BUFFER_SIZE;
	    }
	} else {
	    // Need more data to complete next message
	    break;
	}
    }
}

void
RemoteConnection::writeData()
{
    // should not be called unless there is data to be written
    assert(ringBufferWriteStart!=ringBufferWriteEnd);

    int bytesToWrite;

    if (ringBufferWriteStart<ringBufferWriteEnd) {
	bytesToWrite = ringBufferWriteEnd-ringBufferWriteStart;
    } else {
	bytesToWrite = REMOTE_RING_BUFFER_SIZE-ringBufferWriteStart;
    }
    
    int bytesWritten;
    bytesWritten = write(connectionFD, ringBufferWrite+ringBufferWriteStart,
			 bytesToWrite);

    assert(bytesWritten!=0);

    if (bytesWritten<0) {
	perror("Write failed to client. Need to handle better");
	exit(1);
    } else {
	ringBufferWriteStart += bytesWritten;
	if (bytesWritten==REMOTE_RING_BUFFER_SIZE)
	    bytesWritten = 0;
    }
}
