/**********************************************************************
   Copyright (C) Christopher Yeoh <cyeoh@samba.org> 2005
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**********************************************************************/
#include <assert.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <algorithm>
#include <unistd.h>
#include <fcntl.h>
#include <sys/select.h>
#include <netinet/in.h>
#include <time.h>

#include "Game.H"
#include "GameConfig.H"
#include "NonWizard.H"
#include "protocol.h"
#include "LogFile.H"

Game::Game(int ServerFD)
    : serverFD(ServerFD)
{
    assert(GameConfig::GetConfig("TurnTimeout", &turnTimeout));
    LogFile::Log(LogFile::DEBUG1, "Turn timeout %i\n", turnTimeout);
}

Game::GameState
Game::getGameState() const 
{
    return gameState;
}

void
Game::sendSpellsToAll(const Player::SpellCast &SpellInfo)
{
    map<int, Player *>::iterator player;
    for (player = players.begin(); player != players.end(); player++) {
	player->second->sendSpellCast(SpellInfo);
    }
}

void
Game::addPlayer(int FD, Player *NewPlayer)
{
    assert(players.find(FD)==players.end());
    players[FD] = NewPlayer;
    creatures.push_back(NewPlayer);
}

int
Game::numPlayers() const
{
    return players.size();
}

void
Game::ApplySpells()
{
    map<int, Player *>::iterator player;

    // List of all spells cast
    map<Spell, bool> allSpellsCast;

    // Need to accumlate list of spells on each creature
    // and where they came from
    map<BaseCreature *, vector<Player::SpellCast> > spellsOnTarget;
    
    Spell leftHandSpell, rightHandSpell;
    int leftTargetID, rightTargetID;

    // Keep track of all spells which are cast and create
    // a list of spells which may be applied for each creature
    // in the game
    for (player = players.begin(); player != players.end(); player++) {
	player->second->getSpellsCast(&leftHandSpell, &leftTargetID,
				 &rightHandSpell, &rightTargetID);
	Player::SpellCast spell;
	spell.spellWorked  = false;

	// Accumulate left hand spells
	if (leftHandSpell!=SPL_NONE) {
	    spell.source = player->second;
	    spell.target = BaseCreature::mapIDToPointer(leftTargetID);
	    spell.spellCast = leftHandSpell;
	    
	    allSpellsCast[leftHandSpell] = true;
	    spellsOnTarget[spell.target].push_back(spell); 
	}

	// Accumulate right hand spells 
	if (rightHandSpell!=SPL_NONE) {
	    spell.source = player->second;
	    spell.target = BaseCreature::mapIDToPointer(rightTargetID);
	    spell.spellCast = rightHandSpell;
	    
	    allSpellsCast[rightHandSpell] = true;
	    spellsOnTarget[spell.target].push_back(spell); 
	}
    }


    map<BaseCreature *, vector<Player::SpellCast> >::iterator target;
    vector<Player::SpellCast>::iterator spellToApply;
    vector<BaseCreature *> dispelMagicTarget;
    vector<Player::SpellCast> spellsCast;


    // Resolve spell conflicts
    for (target = spellsOnTarget.begin(); 
	 target != spellsOnTarget.end(); 
	 target++) {
	
	// Dispel magic destroys all spells except other dispel magics
	// Also:
	//   Removes Enchantment on all players
	//   Destroys all monsters (but can still attack this turn)
	//   Acts as shield on target
	//   No effect on stab or surrender
	// First find all dispell magic targets
	if (find_if(target->second.begin(), target->second.end(), 
		    Player::SpellCastPredicateSpellEqual(SPL_DISPEL_MAGIC))
	    != target->second.end()) {
	    dispelMagicTarget.push_back(target->first);
	}

    }

    // 
    string msg;
    if (dispelMagicTarget.size()>0) {
	// Dispel magic cast so remove all enchantments on all 
	// creatures
	vector<BaseCreature *>::iterator creature;
	for (creature=creatures.begin(); creature!=creatures.end(); 
	     creature++) {
	    // This will also mark the creature to be destroyed
	    // as Remove Enchantment does the same thing
	    (*creature)->applyRemoveEnchantmentSpell(&msg);
	}
	
	// Create message noting which spells fizzled?
	// use stateful predicate below to record SpellCast objects
	// removed ? 
	

	// Remove spells affected by dispel magic
 	remove_if(spellsOnTarget[*creature].begin(),
 		  spellsOnTarget[*creature].end(),
 		  Player::SpellCastPredicateSpellEqual(Spells::getSpellsNullifiedByDispellMagic()));
    }


    // Handle counter spell much like dispel magic
    // TODO: implement

    // Magic Mirror
    for (target = spellsOnTarget.begin(); 
	 target != spellsOnTarget.end(); 
	 target++) {

	// See if target has a magic mirror cast on them this round
	if (find_if(target->second.begin(), target->second.end(),
 		 Player::SpellCastPredicateSpellEqual(SPL_MAGIC_MIRROR))
	    !=target->second.end()) {

	    // Reflect spells back at source
	    spellsCast = target->second;
	    for (spellToApply = spellsCast.begin();
		 spellToApply != spellsCast.end();
		 spellToApply++) {
		if (Spells::spellIsReflectable((*spellToApply).spellCast)) {
		    // Check to see if other person also has magic
		    // mirror. If the source has a magic mirror then
		    // spell disappears instead of being reflected.
		    Player *source = (*spellToApply).source;
		    
		    if (find_if(spellsOnTarget[source].begin(),
				spellsOnTarget[source].end(),
				Player::SpellCastPredicateSpellEqual(SPL_MAGIC_MIRROR))
			!=spellsOnTarget[source].end()) {
			// Reflection failed because source has mirror
			// TODO: create message re: failure
			
		    } else {
			// Push spell onto source as magic mirror has reflected
			spellsOnTarget[source].push_back(*spellToApply);
		    }
		}
	    }
	    
	    // Remove reflected spells
	    vector<Player::SpellCast> remainingSpells;
	    for (spellToApply = spellsCast.begin();
		 spellToApply != spellsCast.end();
		 spellToApply++) {
		if (!Spells::spellIsReflectable((*spellToApply).spellCast)) {
		    remainingSpells.push_back(*spellToApply);
		}
	    }
	    // Copies reduced spell list over
	    target->second = remainingSpells;
	}
    }
    
    // Only one of ice-storm or firestorm
    // TODO:

    // Work out which spells cancel each other out
    for (target = spellsOnTarget.begin(); 
	 target != spellsOnTarget.end(); 
	 target++) {

	spellsCast = target->second;

	// Enchantments
	//  amnesia, confusion, charm person, charm monster, paralysis,
	//  fear all cancel each other out
	int enchantment_count = 
	    count_if(spellsCast.begin(),
		     spellsCast.end(),
		     Player::SpellCastPredicateSpellEqual(
		       Spells::getEnchantmentSpellsCancelled()));
	if (enchantment_count>1) {
	    // All them are cancelled
	    Player::SpellCastPredicateSpellEqual 
		pred(Spells::getEnchantmentSpellsCancelled());
	    remove_if(spellsCast.begin(),
		      spellsCast.end(),
		      Player::SpellCastPredicateSpellEqual(pred));
	    vector<Player::SpellCast> removedSpells;
	    vector<Player::SpellCast>::iterator spellCancelled;
	    for (spellCancelled=removedSpells.begin();
		 spellCancelled!=removedSpells.end();
		 spellCancelled++) {
		(*spellCancelled).resultMessage = string("Spell ") + 
		    Spells::getSpellName((*spellCancelled).spellCast) +
		    " on %T by %S failed due to interactions with other "
		    "spells";
		sendSpellsToAll(*spellCancelled);
	    }
	}
							  
									      
	// TODO:

	// fireball, ice-storm
	// firestorem, ice-storm/ice-elemental
	// TODO:
	
    }    

    // Sort spells for each target
    for (target = spellsOnTarget.begin(); 
	 target != spellsOnTarget.end(); 
	 target++) {
	sort(target->second.begin(), target->second.end(), 
	     Player::SpellCastOrderCompare());
    }	 

    // Apply spells on each creature
    for (target = spellsOnTarget.begin(); 
	 target != spellsOnTarget.end();
	 target++) {

	// List of spells on single target
	spellsCast = target->second;
	for (spellToApply = spellsCast.begin();
	     spellToApply != spellsCast.end();
	     spellToApply++) {
	    LogFile::Log(LogFile::DEBUG3, "Applying spell %s",
			 Spells::getSpellName(spellToApply->spellCast));
	    switch (spellToApply->spellCast) {
	    case SPL_SUMMON_GOBLIN:
	    case SPL_SUMMON_OGRE:
	    case SPL_SUMMON_TROLL:
	    case SPL_SUMMON_GIANT:
	    case SPL_SUMMON_ELEMENTAL:
		{
		    NonWizard *monster;
		    Wizard *target 
			= dynamic_cast<Wizard *>(spellToApply->target);

		    // Summoning target must be wizard
		    // Should handle this better? Or just
		    // make sure it never happens
		    assert(target!=NULL);

		    monster = createMonster(spellToApply->spellCast, target);

		    // Assign creature to wizard
		    target->addMonster(monster);
		    spellToApply->resultMessage = "%S summons a %M "
			"who is controlled by %T";
		    spellToApply->spellWorked = true;

		    // Add monster to creature list
		    creatures.push_back(monster);

// 		    sendSpellsToAll(*spellToApply);
// FIXME: Don't need above line as spell result sent to player later anyway
// Is a bit wierd that monster will appear before spell reported
		    map<int, Player *>::iterator player;
		    for (player = players.begin(); player != players.end();
			 player++) {
			player->second->sendNewMonsterInfo(monster);
			player->second->sendCreatureState(monster);
		    }


		    break;
		}
		break;

	    case SPL_CHARM_MONSTER:
		// Will need to call charm monster
		// and reparent to new owner.
		
	    default:
		spellToApply->spellWorked = 
		    target->first->applySpell(spellToApply->spellCast,
					      &spellToApply->resultMessage);
	    }

	    // Send update to all players saying what happened
	    sendSpellsToAll(*spellToApply);
	}
    }
    
    // Send notification to all player saying spells observed effect
    // has ended
    for (player = players.begin(); player != players.end(); player++) {
	player->second->sendSpellCastEnd();
    }
}

void
Game::GetMonsterDirections()
{
    map<int, Player *>::iterator player;
    vector<NonWizard *> monsters;
    vector<NonWizard *>::iterator monster;

    // Work out which creatures are valid targets (visible+alive)
    vector<int> visibleCreatures;
    vector<BaseCreature *>::iterator creature;
    for (creature = creatures.begin(); 
	 creature != creatures.end(); 
	 creature++) {
	if ( !((*creature)->getState() & (CS_DEAD|CS_INVISIBILITY)) ) {
	    visibleCreatures.push_back((*creature)->getID());
	}
    }

    for (player = players.begin(); player != players.end(); player++) {
	monsters = player->second->getMonsters();
	
	// Ask players for monster directions
	player->second->askForMonsterDirections(visibleCreatures);
    }    
}

int
Game::numPlayersAlive()
{
    map<int, Player *>::iterator player;
    int live_count = 0;

    for (player = players.begin(); player != players.end(); player++) {
	if (!player->second->isStateSet(CS_DEAD)) {
	    live_count++;
	}
    }
    return live_count;
}

int
Game::PlayersJoining(time_t ReferenceTime)
{
    fd_set readFs, writeFs;
    struct sockaddr_in sa;
    socklen_t salen;
    int playerFd;
    map<int, Player *>::iterator player;
    int maxFD;
    int startTimeout;
    struct timeval timeout;

    assert(GameConfig::GetConfig("StartTimeout", &startTimeout));

    while (numPlayers()<2 || time(NULL)-ReferenceTime<startTimeout) {
	FD_ZERO(&readFs);
	FD_ZERO(&writeFs);

	// Accepting new connections
	FD_SET(serverFD, &readFs);
	maxFD = serverFD;

	for (player=players.begin(); player!=players.end(); player++) {
	    FD_SET(player->first, &readFs);
	    maxFD = max(maxFD, player->first);

	    if (player->second->connection().pendingDataToWrite()) {
		// Need to send data to player
		FD_SET(player->first, &writeFs);
	    }
	}


	timeout.tv_sec = startTimeout - (time(NULL) - ReferenceTime);
	timeout.tv_usec = 0;

	if (numPlayers()<2) {
	    LogFile::Log(LogFile::DEBUG1, "Waiting for new clients");
	    if (select(maxFD+1, &readFs, &writeFs, NULL, NULL)<0) {
		perror("Select on accepting new clients failed");
		exit(1);
	    }
	} else {
	    LogFile::Log(LogFile::DEBUG1, "Waiting for new clients %li", 
			 timeout.tv_sec);
	    if (select(maxFD+1, &readFs, &writeFs, NULL, &timeout)<0) {
		perror("Select on accepting new clients failed");
		exit(1);
	    }
	}

	if (FD_ISSET(serverFD, &readFs)) {
	    // New player joining 
	    salen = sizeof(sa);
	    playerFd = accept(serverFD, (struct sockaddr *)&sa, &salen);

	    if (playerFd<0) {
		perror("Accept failed for new client");
		exit(1);
	    }

	    // Set connection non blocking
	    int flags;
	    flags = fcntl(playerFd, F_GETFL);
	    if (flags<0) {
		perror("fcntl F_GETFL failed");
		exit(1);
	    }
	    flags = flags | O_NONBLOCK;
	    if (fcntl(playerFd, F_SETFL, flags)<0) {
		perror("failed to make connection non blocking");
		exit(1);
	    }

	    addPlayer(playerFd, new Player(playerFd));

	    ReferenceTime = time(NULL);
	}

	// Check for player player communication
	for (player=players.begin(); player!=players.end(); player++) {
	    if (FD_ISSET(player->first, &readFs)) {
		// Data to read
		player->second->connection().readData();

		// Process messages if any received
		while (player->second->connection().haveMessageToRead()) {
		    player->second->ProcessMessage();
		}
	    }

	    if (FD_ISSET(player->first, &writeFs)) {
		// Can write more data to player
		player->second->connection().writeData();
	    }
	}
    }
    
    return numPlayers();
}

void
Game::StartGame()
{
    LogFile::Log(LogFile::DEBUG1, "Starting Game");

    map<int, Player *>::iterator player;

    assert(numPlayers()>1);

    // synchronise before starting
    WaitForResponses(turnTimeout);

    // Send new player info
    LogFile::Log(LogFile::DEBUG2, "Sending new player info");
    for (player = players.begin(); player != players.end(); player++) {
	map<int, Player *>::iterator newPlayer;
	for (newPlayer = players.begin(); 
	     newPlayer != players.end(); 
	     newPlayer++) {
	    player->second->sendNewPlayerInfo(newPlayer->second);
	}
    }
    
    // Send creature states
    // Send state of all creatures to all players
    vector<BaseCreature *>::iterator creature;
    for (creature = creatures.begin();
	 creature != creatures.end();
	 creature++) {
	for (player = players.begin(); player != players.end(); player++) {
	    player->second->sendCreatureState(*creature);
	}
    }
    // Send end marker
    for (player = players.begin(); player != players.end(); player++) {
	player->second->sendCreatureStateEnd();
    }

    // Keep playing while at least one player is alive
    while (numPlayersAlive()>1) {
	LogFile::Log(LogFile::DEBUG2, 
		     "--------------------------------------------------");
	LogFile::Log(LogFile::DEBUG1, "New round");

	// Get gestures from live players
	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		player->second->askForGestures();
	    }
	}
	LogFile::Log(LogFile::DEBUG2, "Get gestures");
	WaitForResponses(turnTimeout);

	// Apply state which may affect gestures
	// eg paralysis, confusion etc. May need to ask more questions here...

	LogFile::Log(LogFile::DEBUG2, "Calculate spells");
	// Work out what spells have been cast
	vector<Spell> leftHandSpells, rightHandSpells;
	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		// Get spell lists
		player->second->getGestureHistory().getSpells(&leftHandSpells, 
							      &rightHandSpells);
		// Ask players what spells they want to cast
		// (where a hand may cast multiple spells can only cast 1)
 		LogFile::Log(LogFile::DEBUG3, "Player %s %i %i spells",
			     player->second->getName(),
			     leftHandSpells.size(), rightHandSpells.size());
		player->second->askForSpellSelection(leftHandSpells, 
						     rightHandSpells);
	    }
	}

	LogFile::Log(LogFile::DEBUG2, "Get spell lists");
	WaitForResponses(turnTimeout);

	// Send players gestures they have seen
	// modulo players that can't see (looked after by player class)

	// Work out which creatures are valid targets (visible+alive)
	vector<int> visibleCreatures;

	for (creature = creatures.begin(); 
	     creature != creatures.end(); 
	     creature++) {
	    if ( !((*creature)->getState() & (CS_DEAD|CS_INVISIBILITY)) ) {
		visibleCreatures.push_back((*creature)->getID());
	    }
	}

	map<int, Player *>::iterator sourcePlayer;
	for (player = players.begin(); player != players.end(); player++) {
	    // Send gesture updates to dead players
	    for (sourcePlayer = players.begin(); 
		 sourcePlayer != players.end(); 
		 sourcePlayer++) {
		// Only send gestures for live players 
		if (!sourcePlayer->second->isStateSet(CS_DEAD)) {
		    player->second->sendGesturesSeen(sourcePlayer->second);
		}
	    }

	    // Send end marker for gestures seen
	    player->second->sendGesturesSeenEnd();

	    // Ask players where to direct spells
	    if (!player->second->isStateSet(CS_DEAD)) {
		LogFile::Log(LogFile::DEBUG2, 
			     "Asking player for spell directions");
		player->second->askForSpellDirections(visibleCreatures);
	    }
	}

	LogFile::Log(LogFile::DEBUG2, "Get spell directions");
	WaitForResponses(turnTimeout);
	
	// Apply spell effects
	ApplySpells();
	
	// Ask players for who monsters should attack
	LogFile::Log(LogFile::DEBUG2, "Get monster directions");
	GetMonsterDirections();
 	WaitForResponses(turnTimeout);

	// Calculate damage on targets
	map<NonWizard *, int>::const_iterator monster;
	string monsterMsg; 
	int damage;
	map<int, Player *>::iterator infPlayer;

	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		const map<NonWizard *, int> &md 
		    = player->second->getMonsterDirections();
		for (monster = md.begin(); monster != md.end(); monster++) {
		    BaseCreature *target = 
			BaseCreature::mapIDToPointer(monster->second);
		    damage = target->applyMonsterAttack(&monsterMsg, monster->first);
		    for (infPlayer=players.begin(); 
			 infPlayer!=players.end(); infPlayer++) {
			infPlayer->second->sendMonsterAttack(monster->first->getID(),
							     monster->second, 
							     damage, 
							     monsterMsg);
							     
		    }
		}
	    }
	}
	for (infPlayer=players.begin(); infPlayer!=players.end(); 
	     infPlayer++) {
	    infPlayer->second->sendMonsterAttackEnd();
	}

	// "Remove" dead monsters
	// TODO: Need to properly remove elementals.
	// "Remove" dead players??
	// Check if creatures died
	for (creature = creatures.begin();
	     creature != creatures.end();
	     creature++) {
	    if ( (*creature)->getNumHitPoints()<=0) {
		(*creature)->setState(CS_DEAD);
	    }
	}

	// update (eg update counters)
	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		player->second->decrementCounters();
	    }
	}

	// Send state of all creatures to all players
	for (creature = creatures.begin();
	     creature != creatures.end();
	     creature++) {
	    for (player = players.begin(); player != players.end(); player++) {
		player->second->sendCreatureState(*creature);
	    }
	}
	// Send end marker
	for (player = players.begin(); player != players.end(); player++) {
	    player->second->sendCreatureStateEnd();
	}
	LogFile::Log(LogFile::DEBUG1, "End of round");

    }

    if (numPlayersAlive()==0) {
	printf("oops don't handle draws yet!\n");
	exit(1);
    } else {
	// Look for winner
	Player *winner=NULL;
	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		LogFile::Log("Player %s won", player->second->getName());
		winner = player->second;
		break;
	    }
	}
	assert(winner!=NULL);

	// Send notification to all players
	for (player = players.begin(); player != players.end(); player++) {
	    player->second->sendEndGame(winner);
	}
	
    }

}

void
Game::WaitForResponses(int Timeout)
{
    struct timeval timeout;
    timeout.tv_sec = Timeout;
    timeout.tv_usec = 0;

    fd_set readFs, writeFs;
    int maxFD = 0;
    bool waitingForPlayer;
    int numFDReady;
    map<int, Player *>::iterator player;
    while (1) {
	
	waitingForPlayer = false;

	FD_ZERO(&readFs);
	FD_ZERO(&writeFs);

	// Should eventually allow observers, but ignore for now
	// so don't accept new connections

	for (player=players.begin(); player!=players.end(); player++) {
	    FD_SET(player->first, &readFs);
	    maxFD = max(maxFD, player->first);

	    if (player->second->connection().pendingDataToWrite()) {
		// Need to send data to player
		FD_SET(player->first, &writeFs);
		waitingForPlayer = true;
	    }

	    if (player->second->amWaitingForResponse()) {
		waitingForPlayer = true;
	    } 
	}

	if (!waitingForPlayer) {
	    // All players have responded 
	    break;
	}

	numFDReady = select(maxFD+1, &readFs, &writeFs, NULL, &timeout);

	if (numFDReady==0) {
	    LogFile::Log(LogFile::DEBUG1, "Timeout for responses");
	    break;
	} else if (numFDReady<0) {
	    perror("Select on accepting new clients failed");
	    exit(1);
	}

	// Check for player communication
	for (player=players.begin(); player!=players.end(); player++) {
	    if (FD_ISSET(player->first, &readFs)) {
		// Data to read
		player->second->connection().readData();

		// Process messages if any received
		while (player->second->connection().haveMessageToRead()) {
		    player->second->ProcessMessage();
		}
	    }

	    if (FD_ISSET(player->first, &writeFs)) {
		// Can write more data to player
		player->second->connection().writeData();
	    }
	}
    }
}

NonWizard *
Game::createMonster(Spell SummonSpell, Wizard *Owner)
{
    NonWizard *monster;
    static int monsterNum = 0;

    switch (SummonSpell) {
    case SPL_SUMMON_GOBLIN:
	monster = new NonWizard(SC_MONSTER_GOBLIN, 1, 1, Owner);
	break;

    case SPL_SUMMON_OGRE:
	monster = new NonWizard(SC_MONSTER_OGRE, 2, 2, Owner);
	break;

    case SPL_SUMMON_TROLL:
	monster = new NonWizard(SC_MONSTER_TROLL, 3, 3, Owner);
	break;

    case SPL_SUMMON_GIANT:
	monster = new NonWizard(SC_MONSTER_GIANT, 4, 4, Owner);
	break;

    case SPL_SUMMON_ELEMENTAL:
	printf("Summon Elemental not yet implemented\n");
	assert(0);
	break;

    default:
	printf("Not a summoning spell\n");
	assert(0);
    }

    monsterNum++;
    char *monsterName;
    asprintf(&monsterName, "monster%i", monsterNum);
    monster->setName(monsterName);
    free(monsterName);
    
    return monster;
}

