/**********************************************************************
   Copyright (C) Christopher Yeoh <cyeoh@samba.org> 2005
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**********************************************************************/
#include <stdio.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>
#include <unistd.h>
#include <readline/readline.h>
#include <netinet/in.h>

#include "../client/SCClient.H"
#include "../client/protocol.h"


void HandleEventInfo(SCClient *Client)
{
    const SCClient::EventMessage &eventData = Client->getEventMessage();
    printf("%s\n", Client->decodeEventMessage(eventData).c_str());
}

void HandleEndGame(SCClient *Client)
{
    printf("Game has ended\n");
    const vector<int> winnerIDs = Client->getGameWinner();
    if (winnerIDs.size()>1) {
	printf("Game is a draw. There are multiple winners");
    } 

    SCClient::CreatureState cs;
    vector<int>::const_iterator winner;
    for (winner = winnerIDs.begin(); winner != winnerIDs.end();
	 winner ++) {
	Client->getCreatureInfo(*winner, &cs);
	printf("Winner of game is %s\n", cs.name.c_str());
    }
    printf("Exiting.\n");
    exit(0);
}

int PlayerChooseSpellDirection(SCClient *Client, Spell CastSpell, 
			       const vector<int> &Targets)
{
    int index = 1;
    vector<int>::const_iterator target;
    SCClient::CreatureState creature;

    printf("Cast spell %s on:\n", Spells::getSpellName(CastSpell));
    
    for (target=Targets.begin(); target!=Targets.end(); target++) {
	Client->getCreatureInfo(*target, &creature);
	printf("%i - %s\n", index, creature.name.c_str());
	index++;
    }

    int num_chosen;

    while (1) {
	char *answer = readline("Choice (index): ");
	num_chosen = atoi(answer);

	if (num_chosen>0 && num_chosen<index) {
	    return Targets[num_chosen-1];
	}
    }
    

} 

void SendSpellDirections(SCClient *Client)
{
    Spell leftSpell, rightSpell;
    vector<int> validTargets;
    int leftTarget = 0, rightTarget = 0;

    Client->getSpellsCast(&leftSpell, &rightSpell, &validTargets);

    if (leftSpell!=SPL_NONE) {
	leftTarget = PlayerChooseSpellDirection(Client, leftSpell, 
						validTargets);
    }
    if (rightSpell!=SPL_NONE) {
	rightTarget = PlayerChooseSpellDirection(Client, rightSpell,
						 validTargets);
    }

    Client->sendSpellTargets(leftTarget, rightTarget);
}

int
PlayerChooseMonsterDirection(SCClient *Client, int MonsterID,
			     const vector<int> &Targets) 
{
    int index = 1;
    vector<int>::const_iterator target;
    SCClient::CreatureState creature;

    Client->getCreatureInfo(MonsterID, &creature);
    printf("Get monster %s to attack: \n", creature.name.c_str());
    
    for (target=Targets.begin(); target!=Targets.end(); target++) {
	Client->getCreatureInfo(*target, &creature);
	printf("%i - %s\n", index, creature.name.c_str());
	index++;
    }

    int num_chosen;

    while (1) {
	char *answer = readline("Choice (index): ");
	num_chosen = atoi(answer);

	if (num_chosen>0 && num_chosen<index) {
	    return Targets[num_chosen-1];
	}
    }
}

void SendMonsterDirections(SCClient *Client) 
{
    vector<int> validTargets;
    vector<int> monstersToDirect;
    map<int, int> monsterDirections;

    monstersToDirect = Client->getMonstersToDirect(&validTargets);

    vector<int>::iterator monster;
    for (monster = monstersToDirect.begin();
	 monster != monstersToDirect.end();
	 monster++) {
	monsterDirections[*monster] = 
	    PlayerChooseMonsterDirection(Client, *monster, validTargets);
	printf("monster direction %i %i\n", *monster, monsterDirections[*monster]);
    }

    Client->sendMonsterTargets(monsterDirections);
}


Spell PlayerChooseSpell(char *Hand, const vector<Spell> &SpellList) 
{
    printf("Choose spell for %s hand\n", Hand);

    int index = 1;
    vector<Spell>::const_iterator spell;

    for (spell=SpellList.begin(); spell!=SpellList.end(); spell++) {
	printf("%i - %s\n", index, Spells::getSpellName(*spell));
	index++;
    }

    bool validAnswer = false;
    int num_chosen;
    while (!validAnswer) {
	char *answer = readline("Choice (index): ");
	num_chosen = atoi(answer);

	if (num_chosen>0 && num_chosen<index) 
	    validAnswer = true;
    }

    return SpellList[num_chosen-1];
}

void SendSpellChoices(SCClient *Client)
{
    vector<Spell> leftSpells;
    vector<Spell> rightSpells;
    Spell leftSpell, rightSpell;

    Client->getPossibleSpells(&leftSpells, &rightSpells);

    if (leftSpells.size()==0) {
	leftSpell = SPL_NONE;
    } else if (leftSpells.size()==1) {
	leftSpell = leftSpells[0];
    } else {
	leftSpell = PlayerChooseSpell("left", leftSpells);
    }

    if (rightSpells.size()==0) {
	rightSpell = SPL_NONE;
    } else if (rightSpells.size()==1) {
	rightSpell = rightSpells[0];
    } else {
	rightSpell = PlayerChooseSpell("right", rightSpells);
    }

    Client->sendSpellSelection(leftSpell, rightSpell);
}

void PrintCreatureInfo(SCClient *Client, int creatureID) 
{
    SCClient::CreatureState creature;
    Client->getCreatureInfo(creatureID, &creature);

    printf("Name: %s\n", creature.name.c_str());
    printf("Hit Points: %i\n", creature.hitPoints);
    printf("State: %i\n", creature.state);
    if (creature.owner!=0) {
	SCClient::CreatureState owner;
	Client->getCreatureInfo(creature.owner, &owner);
	printf("Owner Name: %s\n", owner.name.c_str());
    }
}

void SendGestures(SCClient *Client)
{
    Gesture leftG, rightG;

    printf("Gestures: \n"
	   "One hand (XX)\n"
	   "F - Fingers spread\n"
	   "P - Palm\n"
	   "S - Snap\n"
	   "W - Wave\n"
	   "D - Point Digit\n"
	   "k - Knife\n"
	   "n - Nothing\n"
	   "Two handed:\n"
	   "c - Clap\n"
	   "d - Double Digit\n"
	   "w - Double Wave\n"
	   "s - Double Snap\n");
    char *line;
    bool validGestures = false;

    while (!validGestures) {
	line = readline("Gestures:");

	if (strlen(line)==1) {
	    validGestures = true;
	    if (line[0]=='c') {
		leftG = rightG = GST_CLAP;
	    } else if (line[0]=='d') {
		leftG = rightG = GST_POINT;
	    } else if (line[0]=='w') {
		leftG = rightG = GST_WAVE;
	    } else if (line[0]=='s') {
		leftG = rightG = GST_SNAP;
	    } else {
		validGestures = false;
	    }
	} else if (strlen(line)==2) {
	    validGestures = true;
	    for (int i=0; i<2; i++) {
		Gesture &gesture = (i==0 ? leftG : rightG);
		switch (line[i]) {

		case 'F':
		    gesture = GST_FINGER;
		    break;

		case 'P':
		    gesture = GST_PALM;
		    break;

		case 'S':
		    gesture = GST_SNAP;
		    break;
		    
		case 'W':
		    gesture = GST_WAVE;
		    break;

		case 'D':
		    gesture = GST_POINT;
		    break;

		case 'k':
		    gesture = GST_KNIFE;
		    if (i==1 && leftG==GST_KNIFE) 
			validGestures = false;
		    break;

		case 'n':
		    gesture = GST_NOTHING;
		    break;
		    
		default:
		    validGestures = false;
		}
	    }
	}
    }
    
    // Should have valid gestures
    Client->sendGestures(leftG, rightG);
}

void ShowWizardGestures(SCClient *Client)
{
    vector<int> creatureIDs;
    vector<int>::iterator creatureID;
    Client->getAllCreatureIDs(&creatureIDs);
    
    for (creatureID=creatureIDs.begin(); creatureID!=creatureIDs.end();
	 creatureID++) {
	SCClient::CreatureState creature;
	Client->getCreatureInfo(*creatureID, &creature);

	if (creature.owner==0) {
	    // Must be wizard
	    Gesture left, right;
	    Client->getGesturesForPlayer(*creatureID, &left, &right);
	    printf("Wizard %s gestured %s (left hand) and %s (right hand)\n",
		   creature.name.c_str(),
		   Spells::getGestureName(left), 
		   Spells::getGestureName(right));
	}
    }
}

void ShowSpellsCast(SCClient *Client)
{
    const vector<SCClient::SpellResult> &spellResults
	 = Client->getSpellResults();
    vector<SCClient::SpellResult>::const_iterator spellResult;

    string source;
    string target;
    SCClient::CreatureState creature;

    for (spellResult = spellResults.begin(); 
	 spellResult != spellResults.end(); spellResult++) {
	Client->getCreatureInfo(spellResult->sourcePlayer, &creature);
	source = creature.name;
	Client->getCreatureInfo(spellResult->targetCreature, &creature);
	target = creature.name;
// 	printf("%s cast %s on %s and it %s (msg %s)\n",
// 	       source.c_str(), Spells::getSpellName(spellResult->spellCast),
// 	       target.c_str(), 
// 	       spellResult->spellWorked ? "worked" : "failed",
// 	       spellResult->resultMessage.c_str());
	if (!spellResult->resultMessage.empty()) {
	    printf("%s\n", Client->decodeMessage(*spellResult).c_str());
	}
    }
}

void ShowMonsterAttackInfo(SCClient *Client)
{
    const vector<SCClient::MonsterAttackResult> mars
	= Client->getMonsterAttackInfo();
    vector<SCClient::MonsterAttackResult>::const_iterator mar;

    SCClient::CreatureState creature;
    string msg;
    
    for (mar = mars.begin(); mar != mars.end(); mar++) {
	printf("%s\n", Client->decodeMessage(*mar).c_str());
    }
}

void Play(SCClient *Client)
{
    // Wait for game start
    int msg;

    while (1) {
	Client->readData();
	while (Client->messageAvailable()) {

	    msg = Client->processMessage();
	    switch (msg) {
	    case MSG_SEND_NEWPLAYER_INFO:
		printf("New player joined\n");
		break;

	    case MSG_ASK_FOR_GESTURES:
		printf("Asked for gestures\n");
 		SendGestures(Client);
		break;

	    case MSG_SEND_GESTURES_SEEN:
		if (Client->haveAllGestures()) {
		    ShowWizardGestures(Client);
		}
		printf("Have been sent gestures\n");
		break;

	    case MSG_SEND_CREATURE_STATE:
		if (Client->haveAllCreatureInfo()) {
		    vector<int> creatures;
		    vector<int>::const_iterator creature;
		    Client->getAllCreatureIDs(&creatures);
		    
		    for (creature=creatures.begin(); creature!=creatures.end();
			 creature++) {
			printf("Creature States:\n");
			PrintCreatureInfo(Client, *creature);
			printf("\n");
		    }
		}
		break;

	    case MSG_ASK_FOR_SPELLS_CAST:
		SendSpellChoices(Client);
		break;

	    case MSG_ASK_SPELL_DIRECTIONS:
		printf("Asked for spell directions\n");
		SendSpellDirections(Client);
		break;


	    case MSG_SEND_SPELL_CAST:
		if (Client->haveAllSpellsCast()) {
		    ShowSpellsCast(Client);
		}
		break;

	    case MSG_SEND_NEW_MONSTER_INFO:
		printf("New monster joined\n");
		break;

	    case MSG_ASK_MONSTER_DIRECTIONS:
		SendMonsterDirections(Client);
		break;

	    case MSG_SEND_MONSTER_ATTACK_INFO:
		if (Client->haveAllMonsterAttackInfo()) {
		    ShowMonsterAttackInfo(Client);
		}
		break;

	    case MSG_SEND_END_GAME:
		HandleEndGame(Client); // Currently exists
		break;

	    case MSG_SEND_EVENT_INFO:
		HandleEventInfo(Client);
		break;

	    default:
		printf("Unhandled message %i received\n", msg);
		;
	    }
	}
    }
}


int main(int argc, char *argv[])
{
    struct sockaddr_in sa;
    struct hostent *server;

    if (argc<2) {
	printf("Usage: %s <username>\n", argv[0]);
	exit(1);
    }
    server = gethostbyname("localhost");
    assert(server);

    int fd = socket(AF_INET, SOCK_STREAM, 0);
    if (fd<0) {
	perror("Could not create socket");
	exit(1);
    }

    memset(&sa, 0, sizeof(struct sockaddr_in)); 
    sa.sin_family = AF_INET;
    sa.sin_port = htons(DEFAULT_SPELLCAST_SERVER_PORT);
    memcpy(&sa.sin_addr.s_addr, server->h_addr_list[0], server->h_length);
    
    if (connect(fd, (struct sockaddr *)&sa, sizeof(sa))<0) {
	perror("Connect failed\n");
	exit(1);
    }

    printf("Connecting with name %s\n", argv[1]);
    SCClient client(fd, argv[1]);

    printf("Server Welcome Message: %s\n", 
	   client.getServerWelcomeMessage().c_str());

    Play(&client);

    exit(1);

}
