/**********************************************************************
   Copyright (C) Christopher Yeoh <cyeoh@samba.org> 2005
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**********************************************************************/
#ifndef _SCCLIENT_H
#define _SCCLIENT_H

#include <map>
#include <string>
#include <vector>

#include "RemoteConnection.H"
#include "Spells.H"

class SCClient {
public:
    SCClient(int Connection, string Name);

    //////////////////////////////////////////////////////////////////////
    // Handling reading/writing to server 

    // Read data from connection
    void readData();

    // New message(s) available
    bool messageAvailable() const;
    
    // Write data
    void writeData();
    // May not be able to write all data
    // Check haveDataToWrite

    // Have data we need to send to server
    bool haveDataToWrite() const;

    int getConnectionFD() const;

    //////////////////////////////////////////////////////////////////////
    
    struct SpellResult {
	int sourcePlayer;
	int targetCreature;
	Spell spellCast;
	bool spellWorked;
	string resultMessage;
    };

    struct MonsterAttackResult {
	int source;
	int target;
	int damage;
	string resultMessage;
    };

    struct CreatureState {
	string name;
	int hitPoints;
	int state;
	int type; // 0 if wizard
	int owner; // 0 if no owner (ie a player)
    };

    struct EventMessage {
	int EventType;
	int MessageSource; // if 0, then from server
	int Source;
	int Target;
	int Misc;
	string MessageString;
    };
    
    // Process a single message from last readData call
    int processMessage();
    // Returns message received
    // Don't call unless you know there is a message ready using
    // messageAvailable
    // Returns the message processed
    // NOTE: All previous state information is invalid after receiving
    // a MSG_SEND_START_GAME

    // Get welcome message sent from server
    const string &getServerWelcomeMessage() const;

    // Get protocol version
    int getProtocolVersion() const;
    
    // Get id of self
    int getOwnID() const;
    // Can call this after receiving a MSG_SEND_START_GAME

    // Get turn timeout
    int getTurnTimeout() const;
    // Wall clock time in seconds that you have to respond to a request
    // Can call this after receiving a MSG_SEND_START_GAME

    // Send gestures to use this turn
    void sendGestures(Gesture LeftHand, Gesture RightHand);
    // You must call this after receiving MSG_ASK_FOR_GESTURES

    struct CreatureState;
    // Get information about monster/player
    void getCreatureInfo(int ID, CreatureState *Creature) const;
    
    // Have received all updated creature information from server
    bool haveAllCreatureInfo() const;

    void getAllCreatureIDs(vector<int> *CreatureIDs) const;

    // Have received all gestures from players
    bool haveAllGestures() const;

    // Get gestures for player
    void getGesturesForPlayer(int ID, Gesture *Left, Gesture *Right);
    // can call anytime after receiving .... (prob need end 
    // of gestures marker

    // Get choices of spells to cast
    void getPossibleSpells(vector<Spell> *LeftHandSpells, 
			   vector<Spell> *RightHandSpells);
    // You can call this after receiving MSG_ASK_FOR_SPELLS_CAST
    // Will only receive MSG_ASK_FOR_SPELLS_CAST if there is a choice
    // to be made (more than two spells can be cast on one hand)

    // Set which spell you wish to cast for each hand
    void sendSpellSelection(Spell LeftHandSpell, Spell RightHandSpell);
    // You must call this after receiving MSG_ASK_FOR_SPELLS_CAST
    // Pass SPL_NONE if no choice for a hand

    // Get spells cast by this player this round (as calculated by server)
    void getSpellsCast(Spell *LeftHandSpell, Spell *RightHandSpell,
		       vector<int> *ValidTargetIDs);
    // Can call this after receiving MSG_ASK_SPELL_DIRECTIONS
    // ValidTargetIDs specifies who you can target spells against
    // but no guarantee that they will work

    // Get spell results
    const vector<SpellResult> &getSpellResults() const;

    // Have received all spells cast
    bool haveAllSpellsCast() const;

    // Get ID of wizard for charm person hand control
    void GetCharmedHandWizard(int *ID) const;
    // Call *immediately* after receiving a MSG_RCV_CHARM_PERSON_CTRL_HAND

    // Send which hand should be controlled
    void SendCharmedHandWizard(int ID, int Hand);
    // Call after receiving a MSG_RCV_CHARM_PERSON_CTRL_HAND

    // Get ID of wizard of hand to be paralysed
    void GetParalysedHandWizard(int *ID) const;
    // Call *immediately* after receiving a MSG_RCV_PARALYSIS_CTRL_HAND

    // Send which hand should be paralysed
    void SendParalysedHandWizard(int ID, int Hand);
    // Call after receiving a MSG_RCV_PARALYSIS_CTRL_HAND

    // Get ID and hand of wizard that is charmed
    void GetCharmedWizard(int *ID, int *Hand) const;
    // Call *immediately* after receiving a MSG_RCV_CHARM_PERSON_CTRL_GESTURE

    // Send gesture for charmed wizard
    void SendCharmedWizard(int ID, Gesture NewGesture);
    // Call after receiving a MSG_RCV_CHARM_PERSON_CTRL_GESTURE

    // Get list of monsters that we control
    const vector<int> &getMonstersToDirect(vector<int> *ValidTargetIDs) const;
    // Can call this after receiving MSG_ASK_MONSTER_DIRECTIONS

    // Send targets for monsters 
    void sendMonsterTargets(const map<int, int> &MonsterTargets);
    // You must call this function after receiving MSG_ASK_MONSTER_DIRECTIONS
    // key is monster ID, value is target ID

    // Get monster attack info
    const vector<MonsterAttackResult> &getMonsterAttackInfo() const;
    
    // Have received all monster attack info
    bool haveAllMonsterAttackInfo() const;

    // Specify who to direct spells against
    void sendSpellTargets(int LeftHandTarget, int RightHandTarget);
    // If there is no valid spell for a hand then ID of 0 is fine.
    // You must call this function after receiving MSG_ASK_SPELL_DIRECTIONS

    // Get who won the game
    const vector<int> &getGameWinner() const;
    // Can call after receiving MSG_SEND_END_GAME
    // Multiple ids returned if it was a draw

    // Decode message for spell result
    string decodeMessage(const SpellResult &SR);
    // Do decoding on client side in case client wants to do 
    // some overriding (eg colourisation)

    // Decode message for monster attack result
    string decodeMessage(const MonsterAttackResult &Mar);

    // Get last event message
    const EventMessage &getEventMessage() const;
    
    // Decode event message
    string decodeEventMessage(const EventMessage &EventData);

private:

    void rcvStartGame(Message *Msg);
    void rcvNewPlayerInfo(Message *Msg);
    void rcvGesturesSeen(Message *Msg);
    void rcvAskSpellDirections(Message *Msg);
    void rcvSpellCast(Message *Msg);
    void rcvCreatureState(Message *Msg);
    void rcvAskForSpellsCast(Message *Msg);
    void rcvAskMonsterDirections(Message *Msg);
    void rcvNewMonsterInfo(Message *Msg);
    void rcvMonsterAttackInfo(Message *Msg);
    void rcvEndGame(Message *Msg);
    void rcvEventMessage(Message *Msg);
    void rcvCharmPersonCtrlHand(Message *Msg);
    void rcvParalysisCtrlHand(Message *Msg);
    void rcvCharmPersonCtrlGesture(Message *Msg);

    RemoteConnection remoteConnection;

    map<int, CreatureState> creatureData; // players + creatures

    string serverWelcomeMessage;

    map<int, Spells::GesturePair> currentGestureMapping;

    vector<int> validTargets;

    vector<SpellResult> spellResults;
    vector<MonsterAttackResult> monsterResults;

    Spell leftSpellCast;
    Spell rightSpellCast;

    vector<Spell> possibleLeftHandSpells;
    vector<Spell> possibleRightHandSpells;
    vector<int> monstersToDirect;
    bool rcvdAllGestures;
    bool rcvdAllSpells;
    bool rcvdAllCreatureInfo;
    bool rcvdAllMonsterResults;

    vector<int> gameWinners;

    EventMessage lastEventMessage;

    int ownID;
    int turnTimeout;
    int protocolVersion;

    int charmTargetID;
    int paralysisTargetID;
    int charmedHand;
};


#endif
