/**********************************************************************
   Copyright (C) Christopher Yeoh <cyeoh@samba.org> 2005
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**********************************************************************/
#include <assert.h>

#include "Spells.H"
#include "LogFile.H"

map< Spell, deque<Spells::GesturePair> > Spells::spellToGestureMap;
vector<Spell> Spells::spellsNullifiedByDispelMagic;
vector<Spell> Spells::spellsNullifiedByCounterSpell;
vector<Spell> Spells::spellsEnchantmentCancelled;

bool Spells::classInitialised = false;

Spells::GesturePair::GesturePair(Gesture Primary, Gesture Secondary)
    : primaryGesture(Primary), secondaryGesture(Secondary)
{
}

Spells::GesturePair::GesturePair()
    : primaryGesture(GST_NOTHING), secondaryGesture(GST_NOTHING)
{
}

void
Spells::init()
{
    deque<GesturePair> gestureList;
    
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    spellToGestureMap[SPL_SHIELD] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    spellToGestureMap[SPL_REMOVE_ENCHANTMENT] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_CLAP, GST_CLAP));
    gestureList.push_back(GesturePair(GST_WAVE, GST_WAVE));
    spellToGestureMap[SPL_MAGIC_MIRROR] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    spellToGestureMap[SPL_COUNTER_SPELL] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    spellToGestureMap[SPL_COUNTER_SPELL1] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_CLAP, GST_CLAP));
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    spellToGestureMap[SPL_DISPEL_MAGIC] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_CLAP, GST_CLAP));
    spellToGestureMap[SPL_RAISE_DEAD] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    spellToGestureMap[SPL_CURE_LIGHT_WOUNDS] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    spellToGestureMap[SPL_CURE_HEAVY_WOUNDS] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    spellToGestureMap[SPL_SUMMON_GOBLIN] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    spellToGestureMap[SPL_SUMMON_OGRE] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    spellToGestureMap[SPL_SUMMON_TROLL] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    spellToGestureMap[SPL_SUMMON_GIANT] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_CLAP, GST_CLAP));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    spellToGestureMap[SPL_SUMMON_FIRE_ELEMENTAL] = gestureList;
    spellToGestureMap[SPL_SUMMON_ICE_ELEMENTAL] = gestureList;
    // fire and ice elemental spells have been separated

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    spellToGestureMap[SPL_MISSILE] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    spellToGestureMap[SPL_FINGER_OF_DEATH] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    spellToGestureMap[SPL_LIGHTNING_BOLT] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_CLAP, GST_CLAP));
    spellToGestureMap[SPL_LIGHTNING_BOLT1] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    spellToGestureMap[SPL_CAUSE_LIGHT_WOUNDS] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    spellToGestureMap[SPL_CAUSE_HEAVY_WOUNDS] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    spellToGestureMap[SPL_FIREBALL] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_CLAP, GST_CLAP));
    spellToGestureMap[SPL_FIRESTORM] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_CLAP, GST_CLAP));
    spellToGestureMap[SPL_ICESTORM] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    spellToGestureMap[SPL_AMNESIA] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    spellToGestureMap[SPL_CONFUSION] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    spellToGestureMap[SPL_CHARM_PERSON] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    spellToGestureMap[SPL_CHARM_MONSTER] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    spellToGestureMap[SPL_PARALYSIS] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    spellToGestureMap[SPL_FEAR] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    spellToGestureMap[SPL_ANTI_SPELL] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    spellToGestureMap[SPL_PROTECTION_FROM_EVIL] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    spellToGestureMap[SPL_RESIST_HEAT] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    spellToGestureMap[SPL_RESIST_COLD] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_CLAP, GST_CLAP));
    spellToGestureMap[SPL_DISEASE] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    spellToGestureMap[SPL_POISON] = gestureList;
    
    gestureList.clear();
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_POINT, GST_POINT));
    spellToGestureMap[SPL_BLINDNESS] = gestureList;
    
    gestureList.clear();
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_WAVE));
    gestureList.push_back(GesturePair(GST_SNAP, GST_SNAP));
    spellToGestureMap[SPL_INVISIBILITY] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_CLAP, GST_CLAP));
    spellToGestureMap[SPL_HASTE] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_CLAP, GST_CLAP));
    spellToGestureMap[SPL_TIME_STOP] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    spellToGestureMap[SPL_DELAYED_EFFECT] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_FINGER, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_PALM, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_SNAP, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_POINT, GST_NOTHING));
    gestureList.push_back(GesturePair(GST_WAVE, GST_NOTHING));
    spellToGestureMap[SPL_PERMANENCY] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_PALM, GST_PALM));
    spellToGestureMap[SPL_SURRENDER] = gestureList;

    gestureList.clear();
    gestureList.push_back(GesturePair(GST_KNIFE, GST_NOTHING));
    spellToGestureMap[SPL_STAB] = gestureList;

    // Spells dispelled by Counter spell
    spellsNullifiedByCounterSpell.push_back(SPL_SHIELD);
    spellsNullifiedByCounterSpell.push_back(SPL_REMOVE_ENCHANTMENT);
    spellsNullifiedByCounterSpell.push_back(SPL_MAGIC_MIRROR);
    spellsNullifiedByCounterSpell.push_back(SPL_RAISE_DEAD);
    spellsNullifiedByCounterSpell.push_back(SPL_CURE_LIGHT_WOUNDS);
    spellsNullifiedByCounterSpell.push_back(SPL_CURE_HEAVY_WOUNDS);
    spellsNullifiedByCounterSpell.push_back(SPL_SUMMON_GOBLIN);
    spellsNullifiedByCounterSpell.push_back(SPL_SUMMON_OGRE);
    spellsNullifiedByCounterSpell.push_back(SPL_SUMMON_TROLL);
    spellsNullifiedByCounterSpell.push_back(SPL_SUMMON_GIANT);
    spellsNullifiedByCounterSpell.push_back(SPL_SUMMON_FIRE_ELEMENTAL);
    spellsNullifiedByCounterSpell.push_back(SPL_SUMMON_ICE_ELEMENTAL);
    spellsNullifiedByCounterSpell.push_back(SPL_MISSILE);
    spellsNullifiedByCounterSpell.push_back(SPL_LIGHTNING_BOLT);
    spellsNullifiedByCounterSpell.push_back(SPL_LIGHTNING_BOLT1);
    spellsNullifiedByCounterSpell.push_back(SPL_CAUSE_LIGHT_WOUNDS);
    spellsNullifiedByCounterSpell.push_back(SPL_CAUSE_HEAVY_WOUNDS);
    spellsNullifiedByCounterSpell.push_back(SPL_FIREBALL);
    spellsNullifiedByCounterSpell.push_back(SPL_FIRESTORM);
    spellsNullifiedByCounterSpell.push_back(SPL_ICESTORM);
    spellsNullifiedByCounterSpell.push_back(SPL_AMNESIA);
    spellsNullifiedByCounterSpell.push_back(SPL_CONFUSION);
    spellsNullifiedByCounterSpell.push_back(SPL_CHARM_PERSON);
    spellsNullifiedByCounterSpell.push_back(SPL_CHARM_MONSTER);
    spellsNullifiedByCounterSpell.push_back(SPL_PARALYSIS);
    spellsNullifiedByCounterSpell.push_back(SPL_FEAR);
    spellsNullifiedByCounterSpell.push_back(SPL_ANTI_SPELL);
    spellsNullifiedByCounterSpell.push_back(SPL_PROTECTION_FROM_EVIL);
    spellsNullifiedByCounterSpell.push_back(SPL_RESIST_HEAT);
    spellsNullifiedByCounterSpell.push_back(SPL_RESIST_COLD);
    spellsNullifiedByCounterSpell.push_back(SPL_DISEASE);
    spellsNullifiedByCounterSpell.push_back(SPL_POISON);
    spellsNullifiedByCounterSpell.push_back(SPL_BLINDNESS);
    spellsNullifiedByCounterSpell.push_back(SPL_INVISIBILITY);
    spellsNullifiedByCounterSpell.push_back(SPL_HASTE);
    spellsNullifiedByCounterSpell.push_back(SPL_TIME_STOP);
    spellsNullifiedByCounterSpell.push_back(SPL_DELAYED_EFFECT);
    spellsNullifiedByCounterSpell.push_back(SPL_PERMANENCY);

    // Spells nullified by dispel magic
    spellsNullifiedByDispelMagic = spellsNullifiedByCounterSpell;
    spellsNullifiedByDispelMagic.push_back(SPL_COUNTER_SPELL);
    spellsNullifiedByDispelMagic.push_back(SPL_COUNTER_SPELL1);
    spellsNullifiedByDispelMagic.push_back(SPL_FINGER_OF_DEATH);

    // Enchantment spells which cancel each other out
    spellsEnchantmentCancelled.push_back(SPL_AMNESIA);
    spellsEnchantmentCancelled.push_back(SPL_CONFUSION);
    spellsEnchantmentCancelled.push_back(SPL_CHARM_PERSON);
    spellsEnchantmentCancelled.push_back(SPL_CHARM_MONSTER);
    spellsEnchantmentCancelled.push_back(SPL_PARALYSIS);
    spellsEnchantmentCancelled.push_back(SPL_FEAR);


    classInitialised = true;
}

const deque<Spells::GesturePair> &
Spells::getGesturesForSpell(const enum Spell Spell)
{
    if (!classInitialised) init();
    return spellToGestureMap[Spell];
}

int
Spells::getSpellsForGestures(const deque<Gesture> &PrimaryHand,
			     const deque<Gesture> &SecondaryHand,
			     vector<Spell> *SpellList)
{
    if (!classInitialised) init();

    deque<Gesture>::const_iterator priGest;
    deque<Gesture>::const_iterator secGest;
    deque<GesturePair>::const_reverse_iterator spellGesturePairs;

    SpellList->clear();

    // Start from 1 as SPL_NONE can't actually be cast
    for (int i=1; i<SPL_FINAL_MARKER; i++) {
	priGest = PrimaryHand.begin();
	secGest = SecondaryHand.begin();

	const deque<GesturePair> spellGestures = spellToGestureMap[(Spell)i];
	spellGesturePairs = spellGestures.rbegin();


	while (priGest != PrimaryHand.end() 
	       && secGest != SecondaryHand.end()) {

	    if (*priGest != spellGesturePairs->primaryGesture
		|| ( (spellGesturePairs->secondaryGesture != GST_NOTHING) 
		     && *secGest != spellGesturePairs->secondaryGesture)) {
		// doesn't match this spell
		break;
	    }
	    
	    priGest++;
	    secGest++;
	    spellGesturePairs++;
	    if (spellGesturePairs == spellGestures.rend()) {
		// Found a spell 
		SpellList->push_back((Spell)i);
		LogFile::Log(LogFile::DEBUG3, "Detected Spell %s", 
			     getSpellName(Spell(i)));
		break;
	    }
	}
    }

    return SpellList->size();
}

const char *
Spells::getSpellName(Spell SpellID) 
{
    switch (SpellID) {
    case SPL_NONE:
	return "None";
    case SPL_SHIELD:
	return "Shield";
    case SPL_REMOVE_ENCHANTMENT:
	return "Remove_enchantment";
    case SPL_MAGIC_MIRROR:
	return "Magic mirror";
    case SPL_COUNTER_SPELL:
    case SPL_COUNTER_SPELL1:
	return "Counter Spell";
    case SPL_DISPEL_MAGIC:
	return "Dispel Magic";
    case SPL_RAISE_DEAD:
	return "Raise dead";
    case SPL_CURE_LIGHT_WOUNDS:
	return "Cure Light Wounds";
    case SPL_CURE_HEAVY_WOUNDS:
	return "Cure Heavy Wounds";
    case SPL_SUMMON_GOBLIN:
	return "Summon Goblin";
    case SPL_SUMMON_OGRE:
	return "Summon Ogre";
    case SPL_SUMMON_TROLL:
	return "Summon Troll";
    case SPL_SUMMON_GIANT:
	return "Summon Giant";
    case SPL_SUMMON_FIRE_ELEMENTAL:
	return "Summon Fire Elemental";
    case SPL_SUMMON_ICE_ELEMENTAL:
	return "Summon Ice Elemental";
    case SPL_MISSILE:
	return "Missile";
    case SPL_FINGER_OF_DEATH:
	return "Finger of Death";
    case SPL_LIGHTNING_BOLT:
    case SPL_LIGHTNING_BOLT1:
	return "Lightning Bolt";
    case SPL_CAUSE_LIGHT_WOUNDS:
	return "Cause light wounds";
    case SPL_CAUSE_HEAVY_WOUNDS:
	return "Cause heavy wounds";
    case SPL_FIREBALL:
	return "Fireball";
    case SPL_FIRESTORM:
	return "Firestorm";
    case SPL_ICESTORM:
	return "Icestorm";
    case SPL_AMNESIA:
	return "Amnesia";
    case SPL_CONFUSION:
	return "Confusion";
    case SPL_CHARM_PERSON:
	return "Charm person";
    case SPL_CHARM_MONSTER:
	return "Charm monster";
    case SPL_PARALYSIS:
	return "Paralysis";
    case SPL_FEAR:
	return "Fear";
    case SPL_ANTI_SPELL:
	return "Anti-spell";
    case SPL_PROTECTION_FROM_EVIL:
	return "Protection from evil";
    case SPL_RESIST_HEAT:
	return "Resist heat";
    case SPL_RESIST_COLD:
	return "Resist cold";
    case SPL_DISEASE:
	return "Disease";
    case SPL_POISON:
	return "Poison";
    case SPL_BLINDNESS:
	return "Blindness";
    case SPL_INVISIBILITY:
	return "Invisibility";
    case SPL_HASTE:
	return "Haste";
    case SPL_TIME_STOP:
	return "Time_STOP";
    case SPL_DELAYED_EFFECT:
	return "Delayed effect";
    case SPL_PERMANENCY:
	return "Permanency";
    case SPL_SURRENDER:
	return "Surrender";
    case SPL_STAB:
	return "stab";
    default:
	return "UNKNOWN SPELL";
    }
}

const char *
Spells::getGestureName(Gesture GestureID)
{
    switch (GestureID) {
    case GST_NOTHING:
	return "nothing";
    case GST_KNIFE:
	return "knife";
    case GST_FINGER:
	return "finger";
    case GST_PALM:
	return "palm";
    case GST_SNAP:
	return "snap";
    case GST_WAVE:
	return "wave";
    case GST_POINT:
	return "point";
    case GST_CLAP:
	return "clap";
    case GST_ANTISPELL:
	return "unknown";
    case GST_FOG:
	return "unknown";
    default:
	return "Unknown Gesture";
    }
}

const vector<Spell> &
Spells::getSpellsNullifiedByDispellMagic() 
{
    return spellsNullifiedByDispelMagic;
}

const vector<Spell> &
Spells::getSpellsNullifiedByCounterSpell() 
{
    return spellsNullifiedByCounterSpell;
}

const vector<Spell> &
Spells::getEnchantmentSpellsCancelled() 
{
    return spellsEnchantmentCancelled;
}

bool
Spells::spellIsReflectable(Spell TestSpell)
{
    switch (TestSpell) {
    case SPL_MAGIC_MIRROR:
    case SPL_STAB:
    case SPL_SURRENDER:
	return false;
	break;

    case SPL_FINAL_MARKER:
    case SPL_NONE:
	assert(0);
	break;

    default:
	return true;
    }
}
