/**********************************************************************
   Copyright (C) Christopher Yeoh <cyeoh@samba.org> 2005
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**********************************************************************/
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>
#include <unistd.h>
#include <netinet/in.h>

#include <qlayout.h>
#include <qscrollview.h>
#include <qmenubar.h>

#include "MainScreen.H"
#include "protocol.h"
#include "StartGameDialog.H"

MainScreen::MainScreen(QWidget *Parent, const char *Name)
    : playerStatus(new PlayerStatus(this, "PlayerStatus")), SC(NULL), 
      currentRequestState(WAITING), initialisedHistoryDisplay(false)

{
    setCaption("Spellcast Client");

    QVBoxLayout *vb = new QVBoxLayout(this);
    playerStatus->setMinimumSize(200,100);
    QHBoxLayout *hb = new QHBoxLayout(this);
    hb->addWidget(playerStatus);

//     QScrollView *scrollView = new QScrollView(this, "GestureHistory");
    ghd = new GestureHistoryDisplay(this);
//     scrollView->addChild(ghd);

    hb->addLayout(ghd);

    playerInput = new PlayerInput(this);

    outputLog = new QTextEdit(this);
    
    vb->addLayout(hb);
    vb->addLayout(playerInput);
    vb->addWidget(outputLog);

    QPopupMenu *popup = new QPopupMenu(this);
    menuBar()->insertItem("Game", popup);
    popup->insertItem("&Connect to Server", this, SLOT(showConnectDialog()));

    connect(playerInput, SIGNAL(itemSelected(int)), 
	    this, SLOT(PlayerMadeSelection(int)));

    for (int i=0; i<GST_ANTISPELL; i++) {
	gestureStringList.push_back(Spells::getGestureName((Gesture(i))));
    }
    handNames.push_back("Left");
    handNames.push_back("Right");
}

void 
MainScreen::LogToUserMsg(const string &Message)
{
    LogToUser(Message.c_str());
}

void 
MainScreen::LogToUser(const QString &Message)
{
    outputLog->append(Message);
}

void
MainScreen::ConnectToHost(const QString &HostName, const QString &Username,
			  bool isObserver)
{
    struct sockaddr_in sa;
    struct hostent *server;

    server = gethostbyname(HostName.latin1());
    assert(server);

    int fd = socket(AF_INET, SOCK_STREAM, 0);
    if (fd<0) {
	perror("Could not create socket");
	exit(1);
    }

    memset(&sa, 0, sizeof(struct sockaddr_in)); 
    sa.sin_family = AF_INET;
    if (isObserver) {
	sa.sin_port = htons(DEFAULT_SPELLCAST_SERVER_OBSERVER_PORT);
    } else {
	sa.sin_port = htons(DEFAULT_SPELLCAST_SERVER_PORT);
    }
    memcpy(&sa.sin_addr.s_addr, server->h_addr_list[0], server->h_length);
    
    if (::connect(fd, (struct sockaddr *)&sa, sizeof(sa))<0) {
	perror("Connect failed\n");
	exit(1);
    }

    amObserver = isObserver;

    SC = new SCClient(fd, Username.latin1());

    LogToUser(SC->getServerWelcomeMessage());

    SCReadNotifier = new QSocketNotifier(SC->getConnectionFD(), 
					 QSocketNotifier::Read, this);
    connect(SCReadNotifier, SIGNAL(activated(int)), 
	    this, SLOT(SCReadyToRead(int)));

    SCWriteNotifier = new QSocketNotifier(SC->getConnectionFD(),
					  QSocketNotifier::Write, this);
    connect(SCWriteNotifier, SIGNAL(activated(int)), 
	    this, SLOT(SCReadyToWrite(int)));
    SCWriteNotifier->setEnabled(SC->haveDataToWrite());
}

void 
MainScreen::SCReadyToRead(int FD)
{
    printf("Enter SCReadyToRead\n");
    assert(FD==SC->getConnectionFD());


    if (SC->messageAvailable()) {
	HandleMessages();
    }

    SC->readData();

    if (SC->messageAvailable()) {
	HandleMessages();
    }
    printf("Exit SCReadyToRead\n");
}

void
MainScreen::AskUserForGestures()
{
    currentRequestState = ASK_LEFT_GESTURE;

    playerInput->SetStateLabel("Enter Left Hand Gesture");
    playerInput->SetList(gestureStringList);
    playerInput->SetConfirmState(false);
}

void
MainScreen::AskUserForSpellChoices()
{
    map<QString, int> spellNames;

    SC->getPossibleSpells(&possibleLeftSpells, &possibleRightSpells);
    if (possibleLeftSpells.size()==0) {
	leftSpell = SPL_NONE;
    } else if (possibleLeftSpells.size()==1) {
	leftSpell = possibleLeftSpells[0];
    } else {
	currentRequestState = ASK_LEFT_SPELL;
	playerInput->SetStateLabel("Select Spell for Left Hand");
	ConvertSpellList(possibleLeftSpells, &spellNames);
	playerInput->SetList(spellNames);
	playerInput->SetConfirmState(false);
	return;
    }
    AskUserForRightSpellChoices();
}

void MainScreen::AskUserForRightSpellChoices()
{
    map<QString, int> spellNames;
    if (possibleRightSpells.size()==0) {
	rightSpell = SPL_NONE;
    } else if (possibleRightSpells.size()==1) {
	rightSpell = possibleRightSpells[0];
	setPlayerInputWait();
    } else {
	currentRequestState = ASK_RIGHT_SPELL;
	playerInput->SetStateLabel("Select Spell for Right Hand");
	ConvertSpellList(possibleRightSpells, &spellNames);
	playerInput->SetList(spellNames);
	playerInput->SetConfirmState(false);
	return;
    }
    SC->sendSpellSelection(leftSpell, rightSpell);
}

void
MainScreen::AskUserForSpellDirections()
{
    vector<int> validTargets;
    vector<int>::const_iterator target;
    SCClient::CreatureState creature;

    SC->getSpellsCast(&leftSpell, &rightSpell, &validTargets);

    // Set targetting to no one initially
    leftTarget = rightTarget = 0;

    // Work out target names
    targetNames.clear();
    for (target=validTargets.begin(); target!=validTargets.end();
	 target++) {
	SC->getCreatureInfo(*target, &creature);
	targetNames[creature.name.c_str()] = *target;
    }

    if (leftSpell!=SPL_NONE) {
	currentRequestState = ASK_LEFT_SPELL_DIRECTION;
	QString spellName = Spells::getSpellName(leftSpell);
	playerInput->SetStateLabel("Select Target for " + spellName + " spell");
	playerInput->SetList(targetNames);
	playerInput->SetConfirmState(false);
	return;
    }

    AskUserForSpellDirectionsRight();
}

void
MainScreen::AskUserForSpellDirectionsRight()
{
    if (rightSpell!=SPL_NONE) {
	currentRequestState = ASK_RIGHT_SPELL_DIRECTION;
	QString spellName = Spells::getSpellName(rightSpell);
	playerInput->SetStateLabel("Select Target for " + spellName + " spell");
	playerInput->SetList(targetNames);
	playerInput->SetConfirmState(false);
	return;
    }

    SC->sendSpellTargets(leftTarget, rightTarget);
}

void
MainScreen::AskUserForMonsterDirections()
{
    vector<int> validTargets;
    SCClient::CreatureState creature;
    vector<int>::const_iterator target;

    monstersToDirect = SC->getMonstersToDirect(&validTargets);

    // Work out target names
    targetNames.clear();
    for (target=validTargets.begin(); target!=validTargets.end();
	 target++) {
	SC->getCreatureInfo(*target, &creature);
	targetNames[creature.name.c_str()] = *target;
    }

    monsterDirections.clear();

    currentRequestState = ASK_MONSTER_DIRECTION;
    monsterToDirect = monstersToDirect.begin();

    SetQueryForMonsterDirection();
}

void
MainScreen::AskForCharmPersonCtrlHand()
{
    int targetID;
    SC->GetCharmedHandWizard(&targetID);

    outstandingHandRequests.push_back(targetID);

    // Check we don't have a current outstanding request
    if (currentRequestState!=ASK_CHOOSE_CHARM_HAND) {
	SetQueryForCharmPersonCtrlHand();
	currentRequestState = ASK_CHOOSE_CHARM_HAND;
    }
}

void
MainScreen::SetQueryForCharmPersonCtrlHand()
{
    currentTarget = outstandingHandRequests.back();
    outstandingHandRequests.pop_back();

    SCClient::CreatureState creature;
    SC->getCreatureInfo(currentTarget, &creature);
    QString monsterName(creature.name.c_str());
    
    playerInput->SetStateLabel("Select hand charmed for " + monsterName);
    playerInput->SetList(handNames);
    playerInput->SetConfirmState(false);
}

void
MainScreen::AskForParalysisCtrlHand()
{
    int targetID;
    SC->GetParalysedHandWizard(&targetID);

    outstandingHandRequests.push_back(targetID);

    // Check we don't have a current outstanding request
    if (currentRequestState!=ASK_CHOOSE_PARALYSIS_HAND) {
	SetQueryForParalysisCtrlHand();
	currentRequestState = ASK_CHOOSE_PARALYSIS_HAND;
    }
}

void
MainScreen::SetQueryForParalysisCtrlHand()
{
    currentTarget = outstandingHandRequests.back();
    outstandingHandRequests.pop_back();

    SCClient::CreatureState creature;
    SC->getCreatureInfo(currentTarget, &creature);
    QString monsterName(creature.name.c_str());
    
    playerInput->SetStateLabel("Select hand paralysed for " + monsterName);
    playerInput->SetList(handNames);
    playerInput->SetConfirmState(false);
}

void
MainScreen::AskForCharmPersonCtrlGesture()
{
    int targetID, hand;
    SC->GetCharmedWizard(&targetID, &hand);

    outstandingHandRequests.push_back(targetID);

    // Check we don't have a current outstanding request
    if (currentRequestState!=ASK_CHOOSE_CHARM_GESTURE) {
	SetQueryForCharmPersonCtrlGesture();
	currentRequestState = ASK_CHOOSE_CHARM_GESTURE;
    }
}

void
MainScreen::SetQueryForCharmPersonCtrlGesture()
{
    currentTarget = outstandingHandRequests.back();
    outstandingHandRequests.pop_back();

    SCClient::CreatureState creature;
    SC->getCreatureInfo(currentTarget, &creature);
    QString monsterName(creature.name.c_str());
    
    playerInput->SetStateLabel("Select gesture for " + monsterName);
    playerInput->SetList(gestureStringList);
    playerInput->SetConfirmState(false);
}

void
MainScreen::SetQueryForMonsterDirection()
{
    SCClient::CreatureState creature;
    SC->getCreatureInfo(*monsterToDirect, &creature);
    QString monsterName(creature.name.c_str());
    
    playerInput->SetStateLabel("Select target for " + monsterName);
    playerInput->SetList(targetNames);
    playerInput->SetConfirmState(false);
}

void
MainScreen::AskUserForNextMonsterDirection()
{
    monsterToDirect++;

    if (monsterToDirect==monstersToDirect.end()) {
	SC->sendMonsterTargets(monsterDirections);
	setPlayerInputWait();
    }
    SetQueryForMonsterDirection();
}

void 
MainScreen::ConvertSpellList(const vector<Spell> &SpellList,
			     map<QString, int> *SpellNames)
{
    SpellNames->clear();
    vector<Spell>::const_iterator spell;
    for (spell=SpellList.begin(); spell!=SpellList.end(); spell++) {
	(*SpellNames)[Spells::getSpellName(*spell)] = *spell;
    }
}

void
MainScreen::ShowSpellsCast()
{
    const vector<SCClient::SpellResult> &spellResults
	 = SC->getSpellResults();
    vector<SCClient::SpellResult>::const_iterator spellResult;

    string source;
    string target;
    SCClient::CreatureState creature;

    for (spellResult = spellResults.begin(); 
	 spellResult != spellResults.end(); spellResult++) {
	SC->getCreatureInfo(spellResult->sourcePlayer, &creature);
	source = creature.name;
	SC->getCreatureInfo(spellResult->targetCreature, &creature);
	target = creature.name;
	if (!spellResult->resultMessage.empty()) {
	    LogToUserMsg(SC->decodeMessage(*spellResult) + "\n");
	}
    }
}

void
MainScreen::ShowEventInfo()
{
    const SCClient::EventMessage &eventData = SC->getEventMessage();
    LogToUserMsg(SC->decodeEventMessage(eventData) + "\n");
}

void
MainScreen::ShowWizardGestures()
{
    vector<int> creatureIDs;
    vector<int>::iterator creatureID;
    QString output;

    SC->getAllCreatureIDs(&creatureIDs);
    
    for (creatureID=creatureIDs.begin(); creatureID!=creatureIDs.end();
	 creatureID++) {
	SCClient::CreatureState creature;
	SC->getCreatureInfo(*creatureID, &creature);

	if (creature.owner==0) {
	    // Must be wizard
	    Gesture left, right;
	    bool antiSpelled;
	    SC->getGesturesForPlayer(*creatureID, &left, &right, &antiSpelled);
	    
	    ghd->AddGestures(*creatureID, left, right);
	    if (antiSpelled) {
		ghd->AddGestures(*creatureID, GST_ANTISPELL, GST_ANTISPELL);
	    }
	    output.sprintf("Wizard %s gestured %s (left hand) and %s (right hand)\n",
			   creature.name.c_str(),
			   Spells::getGestureName(left), 
			   Spells::getGestureName(right));
	    LogToUser(output);
	}
    }
}

void
MainScreen::ShowMonsterAttackInfo()
{
    const vector<SCClient::MonsterAttackResult> mars
	= SC->getMonsterAttackInfo();
    vector<SCClient::MonsterAttackResult>::const_iterator mar;

    SCClient::CreatureState creature;
    string msg;
    
    for (mar = mars.begin(); mar != mars.end(); mar++) {
	LogToUserMsg(SC->decodeMessage(*mar) + "\n");
    }
}

void 
MainScreen::HandleEndGame()
{
    LogToUser("**********************************************");
    LogToUser("Game has ended\n");
    const vector<int> winnerIDs = SC->getGameWinner();
    if (winnerIDs.size()>1) {
	LogToUser("Game is a draw. There are multiple winners");
    } 

    SCClient::CreatureState cs;
    vector<int>::const_iterator winner;
    for (winner = winnerIDs.begin(); winner != winnerIDs.end();
	 winner ++) {
	SC->getCreatureInfo(*winner, &cs);
	LogToUserMsg("Winner of game is " + cs.name + "\n");
    }
    LogToUserMsg("************************************************");
}

void
MainScreen::PlayerMadeSelection(int Selection)
{
    switch (currentRequestState) {
    case WAITING:
	assert(0);
	break;

    case ASK_LEFT_GESTURE:
	leftGesture = Selection;
	currentRequestState = ASK_RIGHT_GESTURE;
	playerInput->SetStateLabel("Enter Right Hand Gesture");
	playerInput->SetList(gestureStringList);
	playerInput->SetConfirmState(false);
	break;
	
    case ASK_RIGHT_GESTURE:
	SC->sendGestures(Gesture(leftGesture), Gesture(Selection));
	setPlayerInputWait();
	break;
	
    case ASK_LEFT_SPELL:
	leftSpell = (Spell) Selection;
	AskUserForRightSpellChoices();
	break;

    case ASK_RIGHT_SPELL:
	rightSpell = (Spell) Selection;
	SC->sendSpellSelection(leftSpell, rightSpell);
	setPlayerInputWait();
	break;
	
    case ASK_LEFT_SPELL_DIRECTION:
	leftTarget = Selection;
	AskUserForSpellDirectionsRight();
	break;

    case ASK_RIGHT_SPELL_DIRECTION:
	rightTarget = Selection;
	SC->sendSpellTargets(leftTarget, rightTarget);
	setPlayerInputWait();
	break;
	
    case ASK_MONSTER_DIRECTION:
	monsterDirections[*monsterToDirect] = Selection;
	AskUserForNextMonsterDirection();
	break;

    case ASK_CHOOSE_CHARM_HAND:
	if (Selection==0) {
	    SC->SendCharmedHandWizard(currentTarget, SC_LEFT_HAND);
	} else {
	    SC->SendCharmedHandWizard(currentTarget, SC_RIGHT_HAND);
	}
	if (outstandingHandRequests.size()>0) {
	    SetQueryForCharmPersonCtrlHand();
	} else {
	    setPlayerInputWait();
	}

	break;

    case ASK_CHOOSE_PARALYSIS_HAND:
	if (Selection==0) {
	    SC->SendParalysedHandWizard(currentTarget, SC_LEFT_HAND);
	} else {
	    SC->SendParalysedHandWizard(currentTarget, SC_RIGHT_HAND);
	}
	if (outstandingHandRequests.size()>0) {
	    SetQueryForParalysisCtrlHand();
	} else {
	    setPlayerInputWait();
	}

	break;
 
    case ASK_CHOOSE_CHARM_GESTURE:
	SC->SendCharmedWizard(currentTarget, (Gesture) Selection);
	if (outstandingHandRequests.size()>0) {
	    SetQueryForCharmPersonCtrlGesture();
	} else {
	    setPlayerInputWait();
	}

	break;
 
   default:
	printf("Unhandled player selection\n");

    };
    SCWriteNotifier->setEnabled(SC->haveDataToWrite());
}

void 
MainScreen::SCReadyToWrite(int )
{
    printf("have data to write\n");
    SCWriteNotifier->setEnabled(false);
    if (SC->haveDataToWrite()) {
	SC->writeData();
    }
    SCWriteNotifier->setEnabled(SC->haveDataToWrite());
}

void
MainScreen::InitialiseHistoryDisplay()
{
    assert(SC->haveAllCreatureInfo());

    vector<int> creatures;
    vector<int>::const_iterator creature;
    SCClient::CreatureState cs;

    SC->getAllCreatureIDs(&creatures);

    for (creature=creatures.begin(); creature!=creatures.end();
	 creature++) {
	SC->getCreatureInfo(*creature, &cs);
	ghd->AddPlayer(cs.name, *creature);
    }
}

void
MainScreen::showConnectDialog()
{
    StartGameDialog *sgd = new StartGameDialog();
    if (sgd->exec()==QDialog::Accepted) {
	ConnectToHost(sgd->GetHostName(), sgd->GetUsername(), 
		      sgd->IsObserver());
    }
}

void
MainScreen::setPlayerInputWait()
{
    vector<QString> emptyList;
    currentRequestState = WAITING;
    playerInput->SetStateLabel("Waiting....");
    playerInput->SetList(emptyList);
    playerInput->SetConfirmState(false);
}

void
MainScreen::HandleMessages()
{
    int msg;
    while (SC->messageAvailable()) {

	msg = SC->processMessage();
	
	printf("Got message %i\n", msg);

	switch (msg) {
	case MSG_SEND_START_GAME:
	    {
		QString gameStart;
		gameStart.sprintf("Game started\nTurn timeout is %i seconds\n",
				  SC->getTurnTimeout());
		LogToUser(gameStart);
		initialisedHistoryDisplay = false;
		ghd->Clear();
	    }
	    break;

	case MSG_SEND_NEWPLAYER_INFO:
	    LogToUser("New player joined\n");
	    break;

 	case MSG_ASK_FOR_GESTURES:
	    LogToUser("Asked for gestures\n");
	    AskUserForGestures();
	    break;

	case MSG_SEND_GESTURES_SEEN:
	    if (SC->haveAllGestures()) {
		ShowWizardGestures();
	    }
	    LogToUser("Have been sent gestures\n");
	    break;

	case MSG_SEND_CREATURE_STATE:
	    if (SC->haveAllCreatureInfo()) {
		playerStatus->updateStatus(SC);
		if (!initialisedHistoryDisplay) {
		    InitialiseHistoryDisplay();
		    initialisedHistoryDisplay = true;
		}
	    }
	    break;

	case MSG_ASK_FOR_SPELLS_CAST:
	    AskUserForSpellChoices();
	    break;

	case MSG_ASK_SPELL_DIRECTIONS:
	    AskUserForSpellDirections();
	    break;


	case MSG_SEND_SPELL_CAST:
	    if (SC->haveAllSpellsCast()) {
		ShowSpellsCast();
	    }
	    break;

	case MSG_SEND_NEW_MONSTER_INFO:
	    LogToUser("New monster joined\n");
	    break;

	case MSG_ASK_MONSTER_DIRECTIONS:
	    AskUserForMonsterDirections();
	    break;

	case MSG_SEND_MONSTER_ATTACK_INFO:
	    if (SC->haveAllMonsterAttackInfo()) {
		ShowMonsterAttackInfo();
	    }
	    break;

	case MSG_SEND_END_GAME:
	    HandleEndGame(); // Currently exists
	    break;

	case MSG_SEND_EVENT_INFO:
	    ShowEventInfo();
	    break;

	case MSG_ASK_CHARM_PERSON_CTRL_HAND:
	    AskForCharmPersonCtrlHand();
	    break;

	case MSG_ASK_PARALYSIS_CTRL_HAND:
	    AskForParalysisCtrlHand();
	    break;

	case MSG_ASK_CHARM_PERSON_CTRL_GESTURE:
	    AskForCharmPersonCtrlGesture();
	    break;

	case MSG_SEND_ROUND_BEGIN:
	    LogToUser("-------------------------------------\n");
	    LogToUser("New Round\n");
	    break;

	default:
	    printf("Unhandled message %i received\n", msg);
	    ;
	}
    }

    SCWriteNotifier->setEnabled(SC->haveDataToWrite());

}
