/**********************************************************************
   Copyright (C) Christopher Yeoh <cyeoh@samba.org> 2005
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**********************************************************************/
#include <assert.h>

#include "BaseCreature.H"
#include "NonWizard.H"
#include "protocol.h"

int BaseCreature::IDCount = 1;
map<int, BaseCreature *> BaseCreature::IDMap;

BaseCreature::BaseCreature(int DamageInflicted, int NumHitPoints)
    : damageInflicted(DamageInflicted), hitPoints(NumHitPoints), 
      maxNumHitPoints(NumHitPoints), state(0), counterProtFromEvil(0),
      counterDisease(0), counterPoison(0), counterHaste(0),
      counterAmnesia(0), counterConfusion(0), counterParalysis(0),
      paralysisController(NULL)
{
    ID = IDCount++;
    IDMap[ID] = this;
}

BaseCreature::~BaseCreature()
{
}

int
BaseCreature::getID() const
{
    return ID;
}

BaseCreature *
BaseCreature::mapIDToPointer(int ID)
{
    assert(ID<IDCount);
    assert(ID!=0);
    return IDMap[ID];
}

bool
BaseCreature::validID(int ID) {
    return ID<IDCount && ID>0;
}

int
BaseCreature::getNumHitPoints() const 
{
    return hitPoints;
}

int 
BaseCreature::getMaxNumHitPoints() const
{
    return maxNumHitPoints;
}

void
BaseCreature::setNumHitPoints(int NumHitPoints)
{
    hitPoints = NumHitPoints;
}

void
BaseCreature::changeNumHitPoints(int Delta)
{
    hitPoints += Delta;
    if (hitPoints > maxNumHitPoints) hitPoints = maxNumHitPoints;
}

void
BaseCreature::setName(const char *Name)
{
    name = Name;
}

const char *
BaseCreature::getName() const
{
    return name.c_str();
}

unsigned int
BaseCreature::getState() const
{
    return state;
}

bool
BaseCreature::isStateSet(int StateBits) const
{
    return state & StateBits;
}

void
BaseCreature::setState(unsigned int State)
{
    state = State;
}

void
BaseCreature::setStateBits(unsigned int StateBits)
{
    state |= StateBits;
}

void
BaseCreature::clearStateBits(unsigned int StateBits)
{
    state &= (StateBits ^ BASE_CREATEURE_STATE_BITS_MASK);
}

int
BaseCreature::getcounterProtFromEvil() const
{
    return counterProtFromEvil;
}

int
BaseCreature::getcounterDisease() const
{
    return counterDisease;
}

int
BaseCreature::getcounterPoison() const
{
    return counterPoison;
}

int
BaseCreature::getDamageInflicted() const
{
    return damageInflicted;
}

bool
BaseCreature::applySpell(Spell SpellToApply, string *Message, 
			 BaseCreature *Source)
{
    switch (SpellToApply) {

    case SPL_SHIELD:
	return applyShieldSpell(Message);
	break;

    case SPL_REMOVE_ENCHANTMENT:
	return applyRemoveEnchantmentSpell(Message);
	break;

    case SPL_MAGIC_MIRROR:
	return applyMagicMirrorSpell(Message);
	break;

    case SPL_COUNTER_SPELL:
    case SPL_COUNTER_SPELL1:
	return applyCounterSpellSpell(Message);
	break;

    case SPL_DISPEL_MAGIC:
	return applyDispelMagicSpell(Message);
	break;
	
    case SPL_RAISE_DEAD:
	return applyRaiseDeadSpell(Message);
	break;

    case SPL_CURE_LIGHT_WOUNDS:
	return applyCureLightWoundsSpell(Message);
	break;

    case SPL_CURE_HEAVY_WOUNDS:
	return applyCureHeavyWoundsSpell(Message);
	break;

    case SPL_SUMMON_GOBLIN:
    case SPL_SUMMON_OGRE:
    case SPL_SUMMON_TROLL:
    case SPL_SUMMON_GIANT:
    case SPL_SUMMON_FIRE_ELEMENTAL:
    case SPL_SUMMON_ICE_ELEMENTAL:
	assert(0);
	break;

    case SPL_MISSILE:
	return applyMissileSpell(Message);
	break;

    case SPL_FINGER_OF_DEATH:
	return applyFingerOfDeathSpell(Message);
	break;

    case SPL_LIGHTNING_BOLT:
    case SPL_LIGHTNING_BOLT1:
	return applyLightningBoltSpell(Message);
	break;

    case SPL_CAUSE_LIGHT_WOUNDS:
	return applyCauseLightWoundsSpell(Message);
	break;

    case SPL_CAUSE_HEAVY_WOUNDS:
	return applyCauseHeavyWoundsSpell(Message);
	break;

    case SPL_FIREBALL:
	return applyFireballSpell(Message);
	break;

    case SPL_FIRESTORM:
	return applyFireStormSpell(Message);
	break;

    case SPL_ICESTORM:
	return applyIceStormSpell(Message);
	break;

    case SPL_AMNESIA:
	return applyAmnesiaSpell(Message);
	break;

    case SPL_CONFUSION:
	return applyConfusionSpell(Message);
	break;

    case SPL_CHARM_PERSON:
	return applyCharmPersonSpell(Message, Source);
	break;

    case SPL_CHARM_MONSTER:
	return applyCharmMonsterSpell(Message, Source);
	break;

    case SPL_PARALYSIS:
	return applyParalysisSpell(Message, Source);
	break;

    case SPL_FEAR:
	return applyFearSpell(Message);
	break;

    case SPL_ANTI_SPELL:
	return applyAntiSpell(Message);
	break;

    case SPL_PROTECTION_FROM_EVIL:
	return applyProtectionFromEvilSpell(Message);
	break;
	
    case SPL_RESIST_HEAT:
	return applyResistHeatSpell(Message);
	break;

    case SPL_RESIST_COLD:
	return applyResistColdSpell(Message);
	break;

    case SPL_DISEASE:
	return applyDiseaseSpell(Message);
	break;
	
    case SPL_POISON:
	return applyPoisonSpell(Message);
	break;

    case SPL_BLINDNESS:
	return applyBlindnessSpell(Message);
	break;

    case SPL_INVISIBILITY:
	return applyInvisibilitySpell(Message);
	break;

    case SPL_HASTE:
	return applyHasteSpell(Message);
	break;

    case SPL_TIME_STOP:
	return applyTimeStopSpell(Message);
	break;

    case SPL_DELAYED_EFFECT:
	return applyDelayedEffectSpell(Message);
	break;

    case SPL_PERMANENCY:
	return applyPermanencySpell(Message);
	break;

    case SPL_SURRENDER:
	applySurrender(Message);
	return true;
	break;

    case SPL_STAB:
	return applyStab(Message);
	break;

    case SPL_FINAL_MARKER:
    case SPL_NONE:
	assert(0);

    }
    return false;
}

bool
BaseCreature::applyShieldSpell(string *Message)
{
    setStateBits(CS_SHIELD);
    *Message = "%S successfully casts Shield spell on %T";
    return true;
}

bool
BaseCreature::applyMagicMirrorSpell(string *Message)
{
    /* Not sure what to do here yet */
    *Message = "%S casts Magic Mirror on %T";
    return true;
}

bool
BaseCreature::applyCounterSpellSpell(string *Message)
{
    /* This spell has other effects which are handled elsewhere,
       but it also doubles as a shield spell, the state of which
       we set here */
    applyShieldSpell(Message);
    *Message = "%S casts Counter Spell on %T";
    return true;
}

bool
BaseCreature::applyRaiseDeadSpell(string *Message)
{
    /* Should raise dead be applied last, or first - impacts
       on what happens here */
    if (getNumHitPoints() <= 0) {
	setNumHitPoints(maxNumHitPoints);
	clearStateBits(ENCHANTMENT_SPELLS);
	clearStateBits(CS_DEAD);
	*Message = "%S raises %T from the dead";
    } else {
	*Message = "%S casts Raise Dead on %T who is already alive. %T is healed 5 hit points";
	changeNumHitPoints(5);
    };
    return true;
}

bool
BaseCreature::applyCureLightWoundsSpell(string *Message)
{
    *Message = "%S casts cure light wounds on %T and heals 1 point of damage";
    changeNumHitPoints(1);
    return true;
}

bool
BaseCreature::applyCureHeavyWoundsSpell(string *Message)
{
    *Message = "%S casts cure heavy wounds on %T and heals 2 points of damage";
    changeNumHitPoints(2);
    clearStateBits(CS_DISEASED);
    return true;
}

bool
BaseCreature::applyMissileSpell(string *Message)
{
    if (!isStateSet(CS_SHIELD)) {
	*Message = "%S casts Missile on %T causing 1 point of damage";
	changeNumHitPoints(-1);
	return true;
    } else {
	*Message = "%S casts Missile on %T but it shatters on %T's shield";
	return false;
    }
}

bool
BaseCreature::applyFingerOfDeathSpell(string *Message)
{
    *Message = "%S casts Finger of Death on %T and kills %T";
    setNumHitPoints(-1000);
    setStateBits(CS_DEAD);
    return true;
}

bool
BaseCreature::applyLightningBoltSpell(string *Message)
{
    *Message = "%S casts a bolt of lighting at %T causing 5 points of damage";
    changeNumHitPoints(-5);
    return true;
}

bool
BaseCreature::applyCauseLightWoundsSpell(string *Message)
{
    *Message = "%S casts Cause Light Wounds on %T causing 2 points of damage";
    changeNumHitPoints(-2);
    return true;
}

bool
BaseCreature::applyCauseHeavyWoundsSpell(string *Message)
{
    *Message = "%S casts Cause Heavy Wounds on %T causing 3 points of damage";
    changeNumHitPoints(-3);
    return true;
}

bool
BaseCreature::applyFireballSpell(string *Message)
{
    if (!isStateSet(CS_RESIST_HEAT)) {
	*Message = "%S casts a fireball at %T causing 5 points of damage";
	changeNumHitPoints(-5);
	return true;
    } else {
	*Message = "%S casts a fireball at %T but %T is resistant to heat";
	return false;
    }
}

bool
BaseCreature::applyFireStormSpell(string *Message)
{
    if (!isStateSet(CS_RESIST_HEAT)) {
	*Message = "Firestorm cast by %S causes 5 points of damage to %T";
	changeNumHitPoints(-5);
	return true;
    } else {
	*Message = "%T is unaffected by the firestorm as %T is resistant to "
	    "heat";
	return false;
    }
}

bool
BaseCreature::applyIceStormSpell(string *Message)
{
    if (!isStateSet(CS_RESIST_COLD)) {
	*Message = "Icestorm cast by %S causes 5 points of damage to %T";
	changeNumHitPoints(-5);
	return true;
    } else {
	*Message = "%T is unaffected by the icestorm as %T is resistant to "
	    "cold";
	return false;
    }
}

bool
BaseCreature::applyAmnesiaSpell(string *Message)
{
    *Message = "%S casts Amnesia on %T";
    counterAmnesia = 2;
    return true;
}

bool 
BaseCreature::applyConfusionSpell(string *Message)
{
    *Message = "%S casts Confusion on %T";
    counterConfusion = 2;
    return true;
}

bool
BaseCreature::applyProtectionFromEvilSpell(string *Message)
{
    setStateBits(CS_PROT_FROM_EVIL);
    setStateBits(CS_SHIELD);
    counterProtFromEvil = 4;
    *Message = "%S casts Protection from Evil on %T";
    return true;
}

bool
BaseCreature::applyResistHeatSpell(string *Message)
{
    *Message = "%S casts Resist Heat on %T";
    setStateBits(CS_RESIST_HEAT);
    return true;
}

bool
BaseCreature::applyResistColdSpell(string *Message)
{
    *Message = "%S casts Resist Cold on %T";
    setStateBits(CS_RESIST_COLD);
    return true;
}

bool
BaseCreature::applyDiseaseSpell(string *Message)
{
    if (!isStateSet(CS_DISEASED)) {
	*Message = "%S casts Disease on %T";
	setStateBits(CS_DISEASED);
	counterDisease = 7;
	return true;
    } else {
	*Message = "%S casts Disease on %T but %T is already diseased";
	return false;
    }
}

bool 
BaseCreature::applyPoisonSpell(string *Message)
{
    if (!isStateSet(CS_POISONED)) {
	*Message = "%S casts Poison on %T";
	setStateBits(CS_POISONED);
	counterPoison = 7;
	return true;
    } else {
	*Message = "%S casts Poison on %T but %T is already poisoned";
	return false;
    }
}

bool
BaseCreature::applyHasteSpell(string *Message)
{
    *Message = "%S casts Haste on %T";
    setStateBits(CS_HASTE);
    counterHaste = 4;
    return true;
}

bool
BaseCreature::applyTimeStopSpell(string *Message)
{
    *Message = "%S casts Time Stop on %T";
    setStateBits(CS_TIMESTOP);
    return true;
}

void
BaseCreature::applySurrender(string *Message)
{
    *Message = "%S surrenders";
    setStateBits(CS_SURRENDER|CS_DEAD); 
    // Can surrended wizards be resurrected?
}

bool
BaseCreature::applyParalysisSpell(string *Message, BaseCreature *Source)
{
    *Message = "%S casts Paralysis on %T";
    counterParalysis = 2;
    paralysisController = Source;
    return true;
}

bool 
BaseCreature::applyStab(string *Message)
{
    if (!isStateSet(CS_SHIELD)) {
	*Message = "%S stabs %T causing 1 point of damage";
	changeNumHitPoints(-1);
	return true;
    } else {
	*Message = "%S attempts to stab %T but is blocked by their shield";
	return false;
    }
}

int
BaseCreature::applyMonsterAttack(string *Message, NonWizard *Monster)
{
    int dmgInflicted = 0;
    if (isStateSet(CS_SHIELD)) {
	*Message = "%S attacks %T but is blocked by a magical shield";
	return dmgInflicted;
    } else if (Monster->getMonsterType()==SC_MONSTER_FIRE_ELEMENTAL
	       && isStateSet(CS_RESIST_HEAT)) {
	*Message = "%S attacks %T but %T is resistant to heat";
	return dmgInflicted;
    } else if (Monster->getMonsterType()==SC_MONSTER_ICE_ELEMENTAL
	       && isStateSet(CS_RESIST_COLD)) {
	*Message = "%S attacks %T but %T is resistant to cold";
	return dmgInflicted;
    } else {
	*Message = "%S attacks %T and does %D points of damage";
	dmgInflicted = Monster->getDamageInflicted();
	changeNumHitPoints(-dmgInflicted);
	return dmgInflicted;
    }
}

void
BaseCreature::decrementCounters() 
{
    if (counterProtFromEvil) {
	counterProtFromEvil--;
	if (counterProtFromEvil==0) {
	    clearStateBits(CS_PROT_FROM_EVIL);
	}
    }


    if (counterDisease) {
	if (!isStateSet(CS_DISEASED)) {
	    counterDisease = 0;
	} else {
	    counterDisease--;
	    if (counterDisease==0) {
		setStateBits(CS_DEAD);
	    }
	}
    }

    if (counterPoison) {
	if (!isStateSet(CS_POISONED)) {
	    counterPoison = 0;
	} else {
	    counterPoison--;
	    if (counterPoison==0) {
		setStateBits(CS_DEAD);
	    }
	}
    }

    if (counterHaste)
	counterHaste--;

    if (!counterProtFromEvil)
	clearStateBits(CS_SHIELD);

    if (counterAmnesia) {
	counterAmnesia--;
	if (counterAmnesia==0) {
	    clearStateBits(CS_AMNESIA);
	} else if (counterAmnesia==1) {
	    // We set the amnesia flag here becaues
	    // it is not initially set by the spell as we don't
	    // want it to affect monsters *this* round, but the next
	    setStateBits(CS_AMNESIA);
	}
    }

    if (counterConfusion) {
	counterConfusion--;
	if (counterConfusion==0) {
	    clearStateBits(CS_CONFUSION);
	} else if (counterConfusion==1) {
	    // We set the confusion flag here becaues
	    // it is not initially set by the spell as we don't
	    // want it to affect monsters *this* round, but the next
	    setStateBits(CS_CONFUSION);
	}
    }

    if (counterParalysis) {
	counterParalysis--;
	if (counterParalysis==0) {
	    clearStateBits(CS_PARALYSIS);
	} else if (counterParalysis==1) {
	    // We set the paralysis flag here becaues
	    // it is not initially set by the spell as we don't
	    // want it to affect monsters *this* round, but the next
	    setStateBits(CS_PARALYSIS);
	}
    }
}

void
BaseCreature::Reset()
{
    hitPoints = maxNumHitPoints;
    state = 0;
    counterProtFromEvil = 0;
    counterDisease = 0;
    counterProtFromEvil = 0;
    counterHaste = 0;
    counterAmnesia = 0;
    counterConfusion = 0;
    counterParalysis = 0;
    paralysisController = NULL;
}

bool
BaseCreature::isParalysedNextTurn() const
{
    return counterParalysis==2;
}

BaseCreature *
BaseCreature::getParalysisController() const
{
    return paralysisController;
}
