/**********************************************************************
   Copyright (C) Christopher Yeoh <cyeoh@samba.org> 2005
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**********************************************************************/
#include <assert.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <algorithm>
#include <unistd.h>
#include <fcntl.h>
#include <sys/select.h>
#include <netinet/in.h>
#include <time.h>

#include "Game.H"
#include "GameConfig.H"
#include "NonWizard.H"
#include "protocol.h"
#include "LogFile.H"
#include "Elemental.H"

Game::Game(int ServerFD, int ObserverFD)
    : serverFD(ServerFD), observerServerFD(ObserverFD)
{
    assert(GameConfig::GetConfig("TurnTimeout", &turnTimeout));
    LogFile::Log(LogFile::DEBUG1, "Turn timeout %i", turnTimeout);

    monsterNames.push_back("Ggorazanoth");
    monsterNames.push_back("Krhanyil");
    monsterNames.push_back("Achichl");
    monsterNames.push_back("Cthac-ekiq");
    monsterNames.push_back("Tha-yiacamal");
    monsterNames.push_back("Yarho");
    monsterNames.push_back("Ihanthus");
    monsterNames.push_back("Of Photuboth");
    monsterNames.push_back("Dhonosand");
    monsterNames.push_back("Chosar");
    monsterNames.push_back("Iquanigud");
    monsterNames.push_back("Yigotello");
    monsterNames.push_back("Dd'gotachasa");
    monsterNames.push_back("Ulegyao");
    monsterNames.push_back("Lek-ila");
    monsterNames.push_back("Homicho-dau");
    monsterNames.push_back("Otsotug-kil");
    monsterNames.push_back("Yitamogh");
    monsterNames.push_back("Urhategot");
    monsterNames.push_back("Ihoggot-pha");
    monsterNames.push_back("Uden-shanogh");
    monsterNames.push_back("Keggosho");
    monsterNames.push_back("Keph-eig");
    monsterNames.push_back("Hugygnat");
    monsterNames.push_back("M'bhazsthagn");
    monsterNames.push_back("Phau-tlehu");

    monsterNameIndex = monsterNames.begin();
}

Game::GameState
Game::getGameState() const 
{
    return gameState;
}

void
Game::sendSpellsToAll(const Player::SpellCast &SpellInfo)
{
    map<int, Player *>::iterator player;
    map<int, Player *>::iterator observer;
    
    for (player = players.begin(); player != players.end(); player++) {
	player->second->sendSpellCast(SpellInfo);
    }
    for (observer = observers.begin(); observer != observers.end(); 
	 observer++) {
	observer->second->sendSpellCast(SpellInfo);
    }
}

void
Game::sendGesturesToAll()
{
    map<int, Player *>::iterator player;
    map<int, Player *>::iterator observer;
    map<int, Player *>::iterator sourcePlayer;

    // Send gestures seen
    for (player = players.begin(); player != players.end(); player++) {
	// Send gesture updates to dead players
	for (sourcePlayer = players.begin(); 
	     sourcePlayer != players.end(); 
	     sourcePlayer++) {
	    // Only send gestures for live players 
	    if (!sourcePlayer->second->isStateSet(CS_DEAD)) {
		player->second->sendGesturesSeen(sourcePlayer->second);
	    }
	}

	// Send end marker for gestures seen
	player->second->sendGesturesSeenEnd();
    }

    for (observer=observers.begin();
	 observer!=observers.end(); observer++) {
	// Send gesture updates to dead players
	for (sourcePlayer = players.begin(); 
	     sourcePlayer != players.end(); 
	     sourcePlayer++) {
	    // Only send gestures for live players 
	    try {
		// Data to read
		if (!sourcePlayer->second->isStateSet(CS_DEAD)) {
		    observer->second->sendGesturesSeen(sourcePlayer->second);
		}
	    } catch (RemoteConnection::ConnectionError Error) {
		ObserverDisconnected(Error);
	    }
	}

	// Send end marker for gestures seen
	observer->second->sendGesturesSeenEnd();
    }
}

void
Game::sendCreatureStateToAll()
{
    map<int, Player *>::iterator player;
    map<int, Player *>::iterator observer;
    vector<BaseCreature *>::iterator creature;

    for (creature = creatures.begin();
	 creature != creatures.end();
	 creature++) {
	for (player = players.begin(); player != players.end(); player++) {
	    player->second->sendCreatureState(*creature);
	}
	for (observer = observers.begin(); observer != observers.end(); 
	     observer++) {
	    observer->second->sendCreatureState(*creature);
	}
    }
    // Send end marker
    for (player = players.begin(); player != players.end(); player++) {
	player->second->sendCreatureStateEnd();
    }
    for (observer = observers.begin(); observer != observers.end(); 
	 observer++) {
	observer->second->sendCreatureStateEnd();
    }
}

void
Game::sendStartRoundToAll(int Round)
{
    map<int, Player *>::iterator player;
    map<int, Player *>::iterator observer;

    for (player = players.begin(); player != players.end(); player++) {
	player->second->sendRoundStart(Round);
    }
    for (observer = observers.begin(); observer != observers.end(); 
	 observer++) {
	observer->second->sendRoundStart(Round);
    }
}

void
Game::sendEndGameToAll(Player *Winner) 
{
    map<int, Player *>::iterator player;
    map<int, Player *>::iterator observer;
    
    for (player = players.begin(); player != players.end(); player++) {
	player->second->sendEndGame(Winner);
    }
    for (observer = observers.begin(); observer != observers.end(); 
	 observer++) {
	observer->second->sendEndGame(Winner);
    }

}
void
Game::sendEndGameToAll(const vector<Player *> &Winners) 
{
    map<int, Player *>::iterator player;
    map<int, Player *>::iterator observer;
    
    for (player = players.begin(); player != players.end(); player++) {
	player->second->sendEndGame(Winners);
    }
    for (observer = observers.begin(); observer != observers.end(); 
	 observer++) {
	observer->second->sendEndGame(Winners);
    }

}

void
Game::addPlayer(int FD, Player *NewPlayer)
{
    assert(players.find(FD)==players.end());
    players[FD] = NewPlayer;
    creatures.push_back(NewPlayer);
}

void
Game::addObserver(int FD, Player *NewObserver)
{
    assert(observers.find(FD)==observers.end());
    observers[FD] = NewObserver;
}

int
Game::numPlayers() const
{
    return players.size();
}

void
Game::ApplySpells()
{
    map<int, Player *>::iterator player;

    // List of all spells cast
    map<Spell, bool> allSpellsCast;

    // Need to accumlate list of spells on each creature
    // and where they came from
    map<BaseCreature *, vector<Player::SpellCast> > spellsOnTarget;
    
    Spell leftHandSpell, rightHandSpell;
    int leftTargetID, rightTargetID;

    // Keep track of all spells which are cast and create
    // a list of spells which may be applied for each creature
    // in the game
    for (player = players.begin(); player != players.end(); player++) {
	if (!player->second->isStateSet(CS_DEAD)) {
	    player->second->getSpellsCast(&leftHandSpell, &leftTargetID,
					  &rightHandSpell, &rightTargetID);
	    Player::SpellCast spell;
	    spell.spellWorked  = false;
	    
	    // Accumulate left hand spells
	    if (leftHandSpell!=SPL_NONE) {
		spell.source = player->second;
		spell.target = BaseCreature::mapIDToPointer(leftTargetID);
		spell.spellCast = leftHandSpell;
		
		allSpellsCast[leftHandSpell] = true;
		spellsOnTarget[spell.target].push_back(spell); 
	    }
	    
	    // Accumulate right hand spells 
	    if (rightHandSpell!=SPL_NONE) {
		spell.source = player->second;
		spell.target = BaseCreature::mapIDToPointer(rightTargetID);
		spell.spellCast = rightHandSpell;
		
		allSpellsCast[rightHandSpell] = true;
		spellsOnTarget[spell.target].push_back(spell); 
	    }
	}
    }

    map<BaseCreature *, vector<Player::SpellCast> >::iterator target;
    vector<Player::SpellCast>::iterator spellToApply;
    vector<BaseCreature *> dispelMagicTarget;
    vector<Player::SpellCast> spellsCast;


    // Resolve spell conflicts
    for (target = spellsOnTarget.begin(); 
	 target != spellsOnTarget.end(); 
	 target++) {
	
	// Dispel magic destroys all spells except other dispel magics
	// Also:
	//   Removes Enchantment on all players
	//   Destroys all monsters (but can still attack this turn)
	//   Acts as shield on target
	//   No effect on stab or surrender
	// First find all dispell magic targets
	if (find_if(target->second.begin(), target->second.end(), 
		    Player::SpellCastPredicateSpellEqual(SPL_DISPEL_MAGIC))
	    != target->second.end()) {
	    dispelMagicTarget.push_back(target->first);
	}
    }

    // 
    string msg;
    if (dispelMagicTarget.size()>0) {
	// Dispel magic cast so remove all enchantments on all 
	// creatures
	vector<BaseCreature *>::iterator creature;
	for (creature=creatures.begin(); creature!=creatures.end(); 
	     creature++) {
	    // This will also mark the creature to be destroyed
	    // as Remove Enchantment does the same thing
	    (*creature)->applyRemoveEnchantmentSpell(&msg);
	    if (dynamic_cast<NonWizard *>(*creature)) {
		SendEventMessageToAllCache(SC_EVT_DISPEL_MAGIC_DESTROY_MONSTER,
					   0, 0, (*creature)->getID(), 0,
					   "%T is destroyed by the Dispel Magic");
	    } else {
		SendEventMessageToAllCache(SC_EVT_DISPEL_MAGIC_DESTROY_MONSTER,
					   0, 0, (*creature)->getID(), 0,
					   "Any enchantments on %T are removed by the Dispel Magic");
	    }
	}
	
	// Create message noting which spells fizzled?
	// use stateful predicate below to record SpellCast objects
	// removed ? 
	vector<Player::SpellCast> dispelled;
	Player::SpellCastPredicateSpellEqual 
	    pred(Spells::getSpellsNullifiedByDispellMagic(), dispelled);

	// Remove spells affected by dispel magic
	vector<Player::SpellCast>::iterator deletedIt = 
	    remove_if(spellsOnTarget[*creature].begin(),
		      spellsOnTarget[*creature].end(),
		      pred);
 	spellsOnTarget[*creature].erase(deletedIt, 
					spellsOnTarget[*creature].end());


	vector<Player::SpellCast>::const_iterator sc;
	for (sc=dispelled.begin(); sc!=dispelled.end(); sc++) {
	    SendEventMessageToAllCache(SC_EVT_DISPEL_MAGIC_NEGATE,
				       0, (*sc).source->getID(),
				       (*sc).target->getID(),
				       (*sc).spellCast,
				       "%S casts %P at %T but it fails due the Dispel Magic");
	}
    }


    {
	// Handle counter spell much like dispel magic
	vector<Player::SpellCast> spellsCountered;
	Player::SpellCastPredicateSpellEqual 
	    pred(Spells::getSpellsNullifiedByCounterSpell(),
		 spellsCountered);
	vector<Spell> counterSpells;
	counterSpells.push_back(SPL_COUNTER_SPELL);
	counterSpells.push_back(SPL_COUNTER_SPELL1);
	for (target = spellsOnTarget.begin(); 
	     target != spellsOnTarget.end();
	     target++) {
	    if (find_if(target->second.begin(), target->second.end(),
			Player::SpellCastPredicateSpellEqual(counterSpells))
		!= target->second.end()) {
		// Wizard has counter spell cast on them
		// Remove all spells that counter spell removes
		LogFile::Log(LogFile::DEBUG1, "Counter spell detected");
		LogFile::Log(LogFile::DEBUG3, "Player %s has %i spells",
			     target->first->getName(),
			     target->second.size());
		pred.clearMatches();
		vector<Player::SpellCast>::iterator deletedIt = 
		    remove_if(target->second.begin(), target->second.end(), pred);
		target->second.erase(deletedIt, target->second.end());

		LogFile::Log(LogFile::DEBUG3, "Player %s how has %i spells\n",
			     target->first->getName(),
			     target->second.size());
	    }
	}

	LogFile::Log(LogFile::DEBUG1, "Counter spell countered %i spells",
		     spellsCountered.size());

	vector<Player::SpellCast>::const_iterator sc;
	for (sc=spellsCountered.begin(); sc!=spellsCountered.end(); sc++) {
	    SendEventMessageToAllCache(SC_EVT_COUNTER_SPELL_NEGATE,
				       0, (*sc).source->getID(),
				       (*sc).target->getID(),
				       (*sc).spellCast,
				       "%S casts %P at %T but it fails due to the counter spell");
	}
    }

    // Magic Mirror
    for (target = spellsOnTarget.begin(); 
	 target != spellsOnTarget.end(); 
	 target++) {

	// See if target has a magic mirror cast on them this round
	if (find_if(target->second.begin(), target->second.end(),
 		 Player::SpellCastPredicateSpellEqual(SPL_MAGIC_MIRROR))
	    !=target->second.end()) {

	    // Reflect spells back at source
	    spellsCast = target->second;
	    for (spellToApply = spellsCast.begin();
		 spellToApply != spellsCast.end();
		 spellToApply++) {
		if (Spells::spellIsReflectable((*spellToApply).spellCast)) {
		    // Check to see if other person also has magic
		    // mirror. If the source has a magic mirror then
		    // spell disappears instead of being reflected.
		    Player *source = (*spellToApply).source;
		    
		    if (find_if(spellsOnTarget[source].begin(),
				spellsOnTarget[source].end(),
				Player::SpellCastPredicateSpellEqual(SPL_MAGIC_MIRROR))
			!=spellsOnTarget[source].end()) {
			SendEventMessageToAllCache(SC_EVT_MAGIC_MIRROR_FAIL,
						   0,
						   spellToApply->source->getID(),
						   spellToApply->target->getID(),
						   spellToApply->spellCast,
						   "%S casts %P at %T which is reflected between their magic mirrors and dissipates");

			// Reflection failed because source has mirror
			// TODO: create message re: failure
			
		    } else if (find_if(spellsOnTarget[source].begin(),
				       spellsOnTarget[source].end(),
				       Player::SpellCastPredicateSpellEqual(SPL_COUNTER_SPELL))
			       != spellsOnTarget[source].end()) {
			SendEventMessageToAllCache(SC_EVT_MAGIC_MIRROR_FAIL,
						   0,
						   spellToApply->source->getID(),
						   spellToApply->target->getID(),
						   spellToApply->spellCast,
						   "%S casts %P at %T which is reflected back to %T by a magic mirror and negated by a counter spell");
		    } else {
			Player::SpellCast newSpell(*spellToApply);
			newSpell.target = source;
			// Push spell onto source as magic mirror has reflected
			spellsOnTarget[source].push_back(newSpell);

			SendEventMessageToAllCache(SC_EVT_MAGIC_MIRROR_SUCCEED,
						   0,
						   spellToApply->source->getID(),
						   spellToApply->target->getID(),
						   spellToApply->spellCast,
						   "%S casts %P at %T which is reflected back at %S");
		    }
		}
	    }
	    
	    // Remove reflected spells
	    vector<Player::SpellCast> remainingSpells;
	    for (spellToApply = spellsCast.begin();
		 spellToApply != spellsCast.end();
		 spellToApply++) {
		if (!Spells::spellIsReflectable((*spellToApply).spellCast)) {
		    remainingSpells.push_back(*spellToApply);
		}
	    }
	    // Copies reduced spell list over
	    target->second = remainingSpells;
	}
    }
    
    // Only one of ice-storm or firestorm
    // Also need to look into existance of ice elementals and fire elementals
    bool haveIceStorm = false, haveFireStorm = false;
    bool haveMultipleIceStorms = false, haveMultipleFireStorms = false;
    Player *fireStormSource = NULL, *iceStormSource = NULL;

    if (allSpellsCast.find(SPL_ICESTORM)!=allSpellsCast.end()) {
	haveIceStorm = true;
    }
    if (allSpellsCast.find(SPL_FIRESTORM)!=allSpellsCast.end()) {
	haveFireStorm = true;
    }

    // Work out which spells cancel each other out
    for (target = spellsOnTarget.begin(); 
	 target != spellsOnTarget.end(); 
	 target++) {

	vector<Player::SpellCast> &sc = target->second;

	// Enchantments
	//  amnesia, confusion, charm person, charm monster, paralysis,
	//  fear all cancel each other out
	int enchantment_count = 
	    count_if(sc.begin(),
		     sc.end(),
		     Player::SpellCastPredicateSpellEqual(
		       Spells::getEnchantmentSpellsCancelled()));
	if (enchantment_count>1) {
	    // All them are cancelled
	    vector<Player::SpellCast> removedSpells;
	    Player::SpellCastPredicateSpellEqual 
		pred(Spells::getEnchantmentSpellsCancelled(),
		     removedSpells);
	    vector<Player::SpellCast>::iterator deletedIt = 
		remove_if(sc.begin(), sc.end(), pred);
	    sc.erase(deletedIt, sc.end());

	    vector<Player::SpellCast>::iterator spellCancelled;
	    for (spellCancelled=removedSpells.begin();
		 spellCancelled!=removedSpells.end();
		 spellCancelled++) {
		(*spellCancelled).resultMessage = string("Spell ") + 
		    Spells::getSpellName((*spellCancelled).spellCast) +
		    " on %T by %S failed due to interactions with other "
		    "spells";
		sendSpellsToAll(*spellCancelled);
	    }
	}

	// Raise dead cancels finger of death
	if (find_if(sc.begin(), sc.end(), 
		 Player::SpellCastPredicateSpellEqual(SPL_RAISE_DEAD))
	    != sc.end() && 
	    find_if(sc.begin(), sc.end(), 
		 Player::SpellCastPredicateSpellEqual(SPL_FINGER_OF_DEATH))
	    !=sc.end()) {

	    // Remove any occurences of either spell on target
	    vector<Spell> spellsToRemove;
	    vector<Player::SpellCast> spellsRemoved;
	    vector<Player::SpellCast>::iterator spellRemoved;

	    spellsToRemove.push_back(SPL_FINGER_OF_DEATH);
	    spellsToRemove.push_back(SPL_RAISE_DEAD);
	    vector<Player::SpellCast>::iterator deletedIt = 
		remove_if(sc.begin(), sc.end(),
			  Player::SpellCastPredicateSpellEqual(spellsToRemove,
							       spellsRemoved));
	    LogFile::Log("before deletion: %i", sc.size());
	    sc.erase(deletedIt, sc.end());
	    LogFile::Log("after deletion: %i", sc.size());

	    for (spellRemoved = spellsRemoved.begin();
		 spellRemoved != spellsRemoved.end();
		 spellRemoved++) {
		SendEventMessageToAllCache(SC_EVT_FOD_CANCELS_RAISE_DEAD,
					   0,
					   (*spellRemoved).source->getID(),
					   (*spellRemoved).target->getID(),
					   (*spellRemoved).spellCast,
					   "%S casts %P cast on %T but Raise Dead cancels Finger of Death");
	    }
	}
    }

    // Sort spells for each target
    for (target = spellsOnTarget.begin(); 
	 target != spellsOnTarget.end(); 
	 target++) {
	sort(target->second.begin(), target->second.end(), 
	     Player::SpellCastOrderCompare());
    }	 

    // Apply spells on each creature
    for (target = spellsOnTarget.begin(); 
	 target != spellsOnTarget.end();
	 target++) {

	// List of spells on single target
	spellsCast = target->second;
	for (spellToApply = spellsCast.begin();
	     spellToApply != spellsCast.end();
	     spellToApply++) {
	    LogFile::Log(LogFile::DEBUG3, "Applying spell %s",
			 Spells::getSpellName(spellToApply->spellCast));
	    switch (spellToApply->spellCast) {
	    case SPL_SUMMON_GOBLIN:
	    case SPL_SUMMON_OGRE:
	    case SPL_SUMMON_TROLL:
	    case SPL_SUMMON_GIANT:
	    case SPL_SUMMON_FIRE_ELEMENTAL:
	    case SPL_SUMMON_ICE_ELEMENTAL:
		{
		    map<int, Player *>::iterator observer;
		    NonWizard *monster;
		    Wizard *wtarget 
			= dynamic_cast<Wizard *>(spellToApply->target);

		    // Summoning target must be wizard
		    // Should handle this better? Or just
		    // make sure it never happens
		    assert(wtarget!=NULL);

		    monster = createMonster(spellToApply->spellCast, wtarget);

		    // Assign creature to wizard
		    wtarget->addMonster(monster);
		    spellToApply->resultMessage = "%S summons a %M "
			"who is controlled by %T";
		    spellToApply->spellWorked = true;

		    // Add monster to creature list
		    creatures.push_back(monster);

// 		    sendSpellsToAll(*spellToApply);
// FIXME: Don't need above line as spell result sent to player later anyway
// Is a bit wierd that monster will appear before spell reported
		    for (player = players.begin(); player != players.end();
			 player++) {
			player->second->sendNewMonsterInfo(monster);
			player->second->sendCreatureState(monster);
		    }
		    for (observer = observers.begin(); 
			 observer != observers.end();
			 observer++) {
			observer->second->sendNewMonsterInfo(monster);
			observer->second->sendCreatureState(monster);
		    }


		    break;
		}
		break;

	    case SPL_FIRESTORM:
		if (haveIceStorm) {
		    // Ice storm cancels firestorm
		    spellToApply->spellWorked = false;
		    spellToApply->resultMessage = "%S casts a Fire Storm but it is cancelled by an Ice Storm\n";
		} else {
		    spellToApply->spellWorked = true;
		    if (!haveMultipleFireStorms) {
			spellToApply->resultMessage = "%S casts a Fire Storm";
			haveMultipleFireStorms = true;
			fireStormSource = spellToApply->source;
		    } else {
			spellToApply->resultMessage = "%S casts a Fire Storm and it merges with the previous storm";
		    }
		}
		break;

	    case SPL_ICESTORM:
		if (haveFireStorm) {
		    // Ice storm cancels firestorm
		    spellToApply->spellWorked = false;
		    spellToApply->resultMessage = "%S casts a Ice Storm but it is cancelled by a Fire Storm\n";
		} else {
		    spellToApply->spellWorked = true;
		    if (!haveMultipleIceStorms) {
			spellToApply->resultMessage = "%S casts an Ice Storm";
			haveMultipleIceStorms = true;
			iceStormSource = spellToApply->source;
		    } else {
			spellToApply->resultMessage = "%S casts an Ice Storm and it merges with the previous Ice Storm";
		    }
		}
		break;

	    default:
		spellToApply->spellWorked = 
		    target->first->applySpell(spellToApply->spellCast,
					      &spellToApply->resultMessage,
					      spellToApply->source);
	    }

	    // Send update to all players saying what happened
	    sendSpellsToAll(*spellToApply);
	}
    }

    // Apply fire storms to all creatures
    if (haveFireStorm && !haveIceStorm) {
	vector<BaseCreature *>::iterator creature;
	Player::SpellCast firestormSpell;
	firestormSpell.source = fireStormSource;
	firestormSpell.spellCast = SPL_FIRESTORM;

	for (creature=creatures.begin();
	     creature!=creatures.end();
	     creature++) {
	    firestormSpell.target = *creature;
	    firestormSpell.spellWorked 
		= (*creature)->applyFireStormSpell(&firestormSpell.resultMessage);
	    sendSpellsToAll(firestormSpell);
	}
    }

    // Apply ice storms to all creatures
    if (haveIceStorm && !haveFireStorm) {
	vector<BaseCreature *>::iterator creature;
	Player::SpellCast icestormSpell;
	icestormSpell.source = iceStormSource;
	icestormSpell.spellCast = SPL_ICESTORM;

	for (creature=creatures.begin();
	     creature!=creatures.end();
	     creature++) {
	    icestormSpell.target = *creature;
	    icestormSpell.spellWorked 
		= (*creature)->applyIceStormSpell(&icestormSpell.resultMessage);
	    sendSpellsToAll(icestormSpell);
	}
    }

    // Send notification to all player saying spells observed effect
    // has ended
    for (player = players.begin(); player != players.end(); player++) {
	player->second->sendSpellCastEnd();
    }
    map<int, Player *>::iterator observer;
    for (observer = observers.begin(); observer != observers.end(); 
	 observer++) {
	observer->second->sendSpellCastEnd();
    }

    // Flush sending of messages to clients
    SendEventMessageToAllCacheFlush();
}

void
Game::GetMonsterDirections()
{
    map<int, Player *>::iterator player;
    vector<NonWizard *> monsters;
    vector<NonWizard *>::iterator monster;

    // Work out which creatures are valid targets (visible+alive)
    vector<int> visibleCreatures;
    vector<BaseCreature *>::iterator creature;
    for (creature = creatures.begin(); 
	 creature != creatures.end(); 
	 creature++) {
	if ( !((*creature)->getState() & (CS_DEAD|CS_INVISIBILITY)) ) {
	    visibleCreatures.push_back((*creature)->getID());
	}
    }

    if (visibleCreatures.size()==0) {
	// Nothing visible, no point asking for monster attack directions
	return;
    }

    for (player = players.begin(); player != players.end(); player++) {
	monsters = player->second->getMonsters();
	for (monster=monsters.begin(); monster!=monsters.end(); monster++) {
	    if ((*monster)->isStateSet(CS_AMNESIA)) {
		SendEventMessageToAll(SC_EVT_PLAYER_AMNESIA, 0, 0,
				      (*monster)->getID(), 0,
				      "%T is forgetfull and attacks the same target as last time");
	    } else if ((*monster)->isStateSet(CS_CONFUSION)) {
		SendEventMessageToAll(SC_EVT_CONFUSION, 0, 0,
				      (*monster)->getID(), 0,
				      "%T is confused and attacks randomly");
	    } else if ((*monster)->isStateSet(CS_PARALYSIS)) {
		SendEventMessageToAll(SC_EVT_MONSTER_PARALYSIS, 0, 0,
				      (*monster)->getID(), 0,
				      "%T is paralysed and doesn't attack");
	    }
	}

	// Ask players for monster directions
	player->second->askForMonsterDirections(visibleCreatures);
    }
}

int
Game::numPlayersAlive()
{
    map<int, Player *>::iterator player;
    int live_count = 0;

    for (player = players.begin(); player != players.end(); player++) {
	if (!player->second->isStateSet(CS_DEAD)) {
	    live_count++;
	}
    }
    return live_count;
}

int
Game::PlayersJoining(time_t ReferenceTime)
{
    fd_set readFs, writeFs;
    struct sockaddr_in sa;
    socklen_t salen;
    map<int, Player *>::iterator player;
    map<int, Player *>::iterator observer;
    int maxFD;
    int startTimeout;
    struct timeval timeout;

    assert(GameConfig::GetConfig("StartTimeout", &startTimeout));

    while (numPlayers()<2 || time(NULL)-ReferenceTime<startTimeout) {
	FD_ZERO(&readFs);
	FD_ZERO(&writeFs);

	// Accepting new connections
	FD_SET(serverFD, &readFs);
	maxFD = serverFD;

	// Accept observers 
	FD_SET(observerServerFD, &readFs);
	maxFD = max(maxFD, observerServerFD);

	for (player=players.begin(); player!=players.end(); player++) {
	    FD_SET(player->first, &readFs);
	    maxFD = max(maxFD, player->first);

	    if (player->second->connection().pendingDataToWrite()) {
		// Need to send data to player
		FD_SET(player->first, &writeFs);
	    }
	}

	for (observer=observers.begin(); 
	     observer!=observers.end(); observer++) {
	    FD_SET(observer->first, &readFs);
	    maxFD = max(maxFD, observer->first);

	    if (observer->second->connection().pendingDataToWrite()) {
		FD_SET(observer->first, &writeFs);
	    }
	}

	time_t currentTime = time(NULL);
	timeout.tv_sec = startTimeout - (currentTime - ReferenceTime);
	timeout.tv_usec = 0;

	if (numPlayers()<2) {
	    LogFile::Log(LogFile::DEBUG1, "Waiting for new clients");
	    if (select(maxFD+1, &readFs, &writeFs, NULL, NULL)<0) {
		perror("Select on accepting new clients failed");
		exit(1);
	    }
	} else {
	    LogFile::Log(LogFile::DEBUG1, "Waiting for new clients %li", 
			 timeout.tv_sec);
	    if (select(maxFD+1, &readFs, &writeFs, NULL, &timeout)<0) {
		perror("Select on accepting new clients failed");
		exit(1);
	    }
	}

	if (FD_ISSET(serverFD, &readFs)) {
	    // New player joining 
	    int playerFd;
	    salen = sizeof(sa);
	    playerFd = accept(serverFD, (struct sockaddr *)&sa, &salen);

	    if (playerFd<0) {
		perror("Accept failed for new client");
		exit(1);
	    }

	    // Set connection non blocking
	    int flags;
	    flags = fcntl(playerFd, F_GETFL);
	    if (flags<0) {
		perror("fcntl F_GETFL failed");
		exit(1);
	    }
	    flags = flags | O_NONBLOCK;
	    if (fcntl(playerFd, F_SETFL, flags)<0) {
		perror("failed to make connection non blocking");
		exit(1);
	    }

	    addPlayer(playerFd, new Player(playerFd, 
					   SPELLCAST_PROTOCOL_VERSION,
					   playerNameList, false));

	    ReferenceTime = time(NULL);
	}

	if (FD_ISSET(observerServerFD, &readFs)) {
	    // New observer joining
	    int observerFD;
	    salen = sizeof(sa);
	    observerFD = 
		accept(observerServerFD, (struct sockaddr *)&sa, &salen);
	    
	    if (observerFD<0) {
		perror("Accept failed for observer");
		LogFile::Log("Observer connection failed");
	    }

	    // Set connection non blocking
	    int flags;
	    flags = fcntl(observerFD, F_GETFL);
	    if (flags<0) {
		perror("fcntl F_GETFL failed");
		exit(1);
	    }
	    flags = flags | O_NONBLOCK;
	    if (fcntl(observerFD, F_SETFL, flags)<0) {
		perror("failed to make connection non blocking");
		exit(1);
	    }

	    addObserver(observerFD, new Player(observerFD, 
					       SPELLCAST_PROTOCOL_VERSION,
					       playerNameList, true));
	}

	// Check for player communication
	for (player=players.begin(); player!=players.end(); player++) {
	    if (FD_ISSET(player->first, &readFs)) {
		
		try {
		    // Data to read
		    player->second->connection().readData();
		    
		    // Process messages if any received
		    while (player->second->connection().haveMessageToRead()) {
			player->second->ProcessMessage();
		    }
		    
		} catch (RemoteConnection::ConnectionError Error) {
		    PlayerDisconnected(Error, true);
		}
	    }
	    
	    if (FD_ISSET(player->first, &writeFs)) {
		try {
		    // Can write more data to player
		    player->second->connection().writeData();
		} catch (RemoteConnection::ConnectionError Error) {
		    PlayerDisconnected(Error, true);
		}
	    }
	}

	for (observer=observers.begin(); 
	     observer!=observers.end(); observer++) {
	    if (FD_ISSET(observer->first, &readFs)) {
		
		try {
		    // Data to read
		    observer->second->connection().readData();
		    
		    // Process messages if any received
		    while (observer->second->connection().haveMessageToRead()) {
			observer->second->ProcessMessage();
		    }
		    
		} catch (RemoteConnection::ConnectionError Error) {
		    ObserverDisconnected(Error);
		}
	    }
	    
	    if (FD_ISSET(observer->first, &writeFs)) {
		try {
		    // Can write more data to player
		    observer->second->connection().writeData();
		} catch (RemoteConnection::ConnectionError Error) {
		    ObserverDisconnected(Error);
		}
	    }
	}
    }
    
    return numPlayers();
}

void
Game::StartGame()
{
    LogFile::Log(LogFile::DEBUG1, "Starting Game");

    map<int, Player *>::iterator player;
    map<int, Player *>::iterator observer;
    int roundCount = 0;

    GameConfig::GetConfig("MaxRoundsPerGame", &maxRoundCount);
    LogFile::Log(LogFile::DEBUG1, "Maximum Rounds per game %i",
		 maxRoundCount);

    assert(numPlayers()>1);

    // synchronise before starting
    WaitForResponses(turnTimeout, true);

    if (numPlayers()<2) {
	LogFile::Log("Game aborted - players must have disconnected (%i)",
		     numPlayers());
	return;
    }

    // Send new player info
    LogFile::Log(LogFile::DEBUG2, "Sending new player info");
    for (player = players.begin(); player != players.end(); player++) {
	// Send start game status
	player->second->sendStartGame(turnTimeout, maxRoundCount);
	map<int, Player *>::iterator newPlayer;
	for (newPlayer = players.begin(); 
	     newPlayer != players.end(); 
	     newPlayer++) {
	    player->second->sendNewPlayerInfo(newPlayer->second);
	}
    }
    for (observer = observers.begin(); observer != observers.end(); 
	 observer++) {
	// Send start game status
	observer->second->sendStartGame(turnTimeout, maxRoundCount);
	map<int, Player *>::iterator newPlayer;
	for (newPlayer = players.begin(); 
	     newPlayer != players.end(); 
	     newPlayer++) {
	    observer->second->sendNewPlayerInfo(newPlayer->second);
	}
    }
    
    // Send creature states
    // Send state of all creatures to all players
    sendCreatureStateToAll();

    vector<Player *> roundStartPlayers;

    // Keep playing while at least one player is alive
    while (numPlayersAlive()>1 && roundCount<maxRoundCount) {
	LogFile::Log(LogFile::DEBUG2, 
		     "--------------------------------------------------");
	LogFile::Log(LogFile::DEBUG1, "New round");

	// Keep track of players alive now in case there are none
	// alive by the end
	roundStartPlayers.clear();
	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		roundStartPlayers.push_back(player->second);
	    }
	}

	// Send message saying round has started
	sendStartRoundToAll(roundCount);

	// Get gestures from live players
	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		if (player->second->isStateSet(CS_FEAR)) {
		    SendEventMessageToAll(SC_EVT_PLAYER_FEAR,
					  0, 0, player->second->getID(), 0,
					  "%T is afraid and can't make all gestures this round");
		}
		
		if (player->second->isStateSet(CS_AMNESIA)) {
		    // Would repeat last gestures
		    SendEventMessageToAll(SC_EVT_PLAYER_AMNESIA,
					  0, 0, player->second->getID(), 0,
					  "%T is forgetful and repeats the previous gestures");
		    player->second->repeatLastGestures();
		} else {
		    player->second->askForGestures();
		}
	    }
	}
	LogFile::Log(LogFile::DEBUG2, "Get gestures");
	WaitForResponses(turnTimeout);

	// Apply state which may affect gestures
	// eg paralysis, confusion etc. May need to ask more questions here...
	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		if (player->second->isStateSet(CS_CONFUSION)) {
		    // Need to randomly change one hand to different gesture
		    Gesture newGesture;
		    switch (random()%6) {
		    case 0:
			newGesture = GST_CLAP;
			break;

		    case 1:
			newGesture = GST_PALM;
			break;

		    case 2:
			newGesture = GST_POINT;
			break;

		    case 3:
			newGesture = GST_FINGER;
			break;

		    case 4:
			newGesture = GST_SNAP;
			break;

		    case 5:
			newGesture = GST_WAVE;
			break;

		    default:
			assert(0);
		    }
		    if (random()%2) {
			player->second->setLeftGesture(newGesture);
		    } else {
			player->second->setRightGesture(newGesture);
		    }
		    SendEventMessageToAll(SC_EVT_CONFUSION,
					  0, 0, player->second->getID(), 0,
					  "%T is confused and fumbles a gesture");
		}
		if (player->second->isStateSet(CS_CHARM_PERSON)) {
		    // Need to ask controller what gesture the player
		    // should make
		    Player *controller = dynamic_cast<Player *>
			(player->second->getCharmer());
		    assert(controller);
		    // Charm does not work if controller has died
		    if (!controller->isStateSet(CS_DEAD)) {
			controller->askCharmedGesture(player->second);
		    }
		}
		if (player->second->isStateSet(CS_PARALYSIS)) {
		    player->second->changeGestureForParalysis();
		}
	    }
	}

	LogFile::Log(LogFile::DEBUG1, 
		     "Wait for charm person gesture responses");
	WaitForResponses(turnTimeout);

	LogFile::Log(LogFile::DEBUG2, "Calculate spells");
	// Work out what spells have been cast
	vector<Spell> leftHandSpells, rightHandSpells;
	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		// Get spell lists
		player->second->getGestureHistory().getSpells(&leftHandSpells, 
							      &rightHandSpells);
		// Can only cast short version of lightning spell once
		if (player->second->haveUsedShortLightning()) {
		    vector<Spell>::iterator lb =
			find(leftHandSpells.begin(), leftHandSpells.end(),
			     SPL_LIGHTNING_BOLT1);
		    if (lb!=leftHandSpells.end()) {
			leftHandSpells.erase(lb);
			LogFile::Log(LogFile::DEBUG1, 
				     "Removed lightning bolt from lh because already used short version of spell");
		    }
		    
		    lb = find(rightHandSpells.begin(), rightHandSpells.end(),
			      SPL_LIGHTNING_BOLT1);
		    if (lb!=rightHandSpells.end()) {
			rightHandSpells.erase(lb);
			LogFile::Log(LogFile::DEBUG1, 
				     "Removed lightning bolt from rh because already used short version of spell");
		    }
		}


		// Ask players what spells they want to cast
		// (where a hand may cast multiple spells can only cast 1)
 		LogFile::Log(LogFile::DEBUG3, "Player %s %i %i spells",
			     player->second->getName(),
			     leftHandSpells.size(), rightHandSpells.size());
		player->second->askForSpellSelection(leftHandSpells, 
						     rightHandSpells);
	    }
	}

	LogFile::Log(LogFile::DEBUG2, "Get spell lists");
	WaitForResponses(turnTimeout);

	// Send players gestures they have seen
	// modulo players that can't see (looked after by player class)

	// Work out which creatures are valid targets (visible+alive)
	vector<int> visibleCreatures;
	vector<int> allCreatures;
	vector<BaseCreature *>::iterator creature;

	for (creature = creatures.begin(); 
	     creature != creatures.end(); 
	     creature++) {
	    if ( !((*creature)->getState() & (CS_DEAD|CS_INVISIBILITY)) ) {
		visibleCreatures.push_back((*creature)->getID());
	    }
	    allCreatures.push_back((*creature)->getID());
	}

	// Ask players for where to direct spells
	for (player = players.begin(); player != players.end(); player++) {
	    // Ask players where to direct spells
	    if (!player->second->isStateSet(CS_DEAD)) {
		LogFile::Log(LogFile::DEBUG2, 
			     "Asking player for spell directions");
		player->second->askForSpellDirections(visibleCreatures,
						      allCreatures);
	    }
	}

	LogFile::Log(LogFile::DEBUG2, "Get spell directions");
	WaitForResponses(turnTimeout);
	
	// Apply spell effects
	ApplySpells();

	LogFile::Log(LogFile::DEBUG1, "Wait for charm person hand responses");
	WaitForResponses(turnTimeout);

	// Send out charm person hand control information
	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		if (player->second->isStateSet(CS_CHARM_PERSON)) {
		    SendEventMessageToAll(SC_EVT_CHARMED_HAND, 0,
					  player->second->getCharmer()->getID(),
					  player->second->getID(),
					  player->second->getHandCharmed(),
					  "%T has their %H hand controlled by %S next round");
		}
	    }
	}

	sendGesturesToAll();

	// Ask about which hand to paralyze
	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		// If player was already paralysed, not choice about hand
		// which is paralysed next turn
		if (player->second->isParalysedNextTurn() &&
		    !player->second->isStateSet(CS_PARALYSIS)) {
		    Player *controller = dynamic_cast<Player *>
			(player->second->getParalysisController());
		    assert(controller!=NULL);
		    controller->askForHandToParalyze(player->second);
		}
	    }
	}

	LogFile::Log(LogFile::DEBUG1, "Wait for paralysis responses");
	WaitForResponses(turnTimeout);


	// Notify players of which hand was paralysed
	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		if (player->second->isParalysedNextTurn()) {
		    SendEventMessageToAll(SC_EVT_PARALYSED_HAND,
					  0, 0, player->second->getID(),
					  player->second->getHandParalysed(),
					  "%T has their %H hand paralysed");
		}
	    }
	}



	///////////////////////////////////////////////////////////////
	// Monster handling

	// Elemental negation
	// If both an ice elemental and fire element exist then they cancel
	// each other out
	bool haveBothElementals = false;
	vector<Elemental *> iceElementals, fireElementals;
	// Search for elementals
	for (player = players.begin(); 
	     player != players.end() && !haveBothElementals; 
	     player++) {
	    
	    const vector<NonWizard *> &monsters = player->second->getMonsters();
	    vector<NonWizard *>::const_iterator monster;
	    Elemental *elemental;
	    
	    for (monster=monsters.begin(); monster!=monsters.end();
		 monster++) {
		switch ((*monster)->getMonsterType()) {
		case SC_MONSTER_FIRE_ELEMENTAL:
		    elemental = dynamic_cast<Elemental *>(*monster);
		    assert(elemental);
		    fireElementals.push_back(elemental);
		    break;

		case SC_MONSTER_ICE_ELEMENTAL:
		    elemental = dynamic_cast<Elemental *>(*monster);
		    assert(elemental);
		    iceElementals.push_back(elemental);
		    break;
		}
	    }
	}

	// Merge elementals
	if (iceElementals.size()>1) {
	    for (unsigned int i=1; i<iceElementals.size(); i++) {
		iceElementals[i]->setStateBits(CS_DEAD);
		SendEventMessageToAll(SC_EVT_ELEMENTAL_MERGE,
				      0,
				      iceElementals[i]->getID(),
				      iceElementals[0]->getID(), 0,
				      "Ice Elemental %S merges with %T");
	    }
	}
	if (fireElementals.size()>1) {
	    for (unsigned int i=1; i<fireElementals.size(); i++) {
		fireElementals[i]->setStateBits(CS_DEAD);
		SendEventMessageToAll(SC_EVT_ELEMENTAL_MERGE,
				      0,
				      fireElementals[i]->getID(),
				      fireElementals[0]->getID(), 0,
				      "Fire Elemental %S merges with %T");
	    }
	}

	if (iceElementals.size()>0 && fireElementals.size()>0) {
	    // Need to destroy all elementals
	    // Only one of each will be left by this stage
	    SendEventMessageToAll(SC_EVT_ELEMENTAL_CANCEL, 0,
				  iceElementals[0]->getID(),
				  fireElementals[0]->getID(), 0,
				  "%S and %T destroy each other");
		iceElementals[0]->setStateBits(CS_DEAD);
		fireElementals[0]->setStateBits(CS_DEAD);
	}
	
	// FIXME: need to update state for elementals which died
	// or rely on client working it out?

	// Ask players for who monsters should attack
	LogFile::Log(LogFile::DEBUG2, "Get monster directions");
	GetMonsterDirections();
 	WaitForResponses(turnTimeout);

	// Calculate damage on targets
	map<NonWizard *, int>::const_iterator monster;
	Elemental *elemental;

	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		const map<NonWizard *, int> &md 
		    = player->second->getMonsterDirections();
		for (monster = md.begin(); monster != md.end(); monster++) {
		    elemental = dynamic_cast<Elemental *>(monster->first);
		    if (elemental) {
			// Attack everyone (except self)
			for (creature=creatures.begin();
			     creature!=creatures.end(); 
			     creature++) {
			    if (!(*creature)->isStateSet(CS_DEAD)
				&& dynamic_cast<Elemental *>(*creature)
				!= elemental) {
				monsterAttack(elemental, *creature);
			    }
			}
		    } else {
			// If id of target is 0, then not attacking anyone
			// eg paralysis, etc
			if (monster->second!=0) {
			    BaseCreature *target = 
				BaseCreature::mapIDToPointer(monster->second);
			    monsterAttack(monster->first, target);
			} else {
			    LogFile::Log(LogFile::DEBUG2, "Monster %s is not targetted at anyone",
					 monster->first->getName());
			}
		    };
		}
	    }
	}

	for (player=players.begin(); player!=players.end(); 
	     player++) {
	    player->second->sendMonsterAttackEnd();
	}
	for (observer=observers.begin(); observer!=observers.end();
	     observer++) {
	    observer->second->sendMonsterAttackEnd();
	}

	// Check if creatures died
	for (creature = creatures.begin();
	     creature != creatures.end();
	     creature++) {
	    if (!(*creature)->isStateSet(CS_DEAD)) {
		// Check to see if creature died due to lack
		// of hit points
		if ( (*creature)->getNumHitPoints()<=0) {
		    SendEventMessageToAll(SC_EVT_PLAYER_DIED, 0,
					  0, (*creature)->getID(), 0,
					  "Player %T has died");
		    (*creature)->setStateBits(CS_DEAD);
		}

		// Update state of creature
		(*creature)->decrementCounters();
	    }
	}

	for (creature = creatures.begin();
	     creature != creatures.end();
	     creature++) {
	    // If its a monster and its owner is dead, then
	    // monster also dies
	    NonWizard *checkMonster = dynamic_cast<NonWizard *>(*creature);
	    if (checkMonster && !checkMonster->isStateSet(CS_DEAD) 
		&& checkMonster->getOwner()
		&& checkMonster->getOwner()->isStateSet(CS_DEAD)) {
		checkMonster->setStateBits(CS_DEAD);
		checkMonster->setNumHitPoints(0);
		SendEventMessageToAll(SC_EVT_NO_OWNER_MONSTER_DIES, 0, 0,
				      checkMonster->getID(), 0,
				      "%T dies when his owner dies");
	    }
	}

	

	// Send state of all creatures to all players
	sendCreatureStateToAll();
	LogFile::Log(LogFile::DEBUG1, "End of round");
	roundCount++;
    }

    if (numPlayersAlive()==0) {
	// Send notification to all players
	LogFile::Log("Drawn game");
	vector<Player *>::iterator winner;
	for (winner = roundStartPlayers.begin(); 
	     winner != roundStartPlayers.end(); winner++) {
	    LogFile::Log("++++Player %s draw", (*winner)->getName());
	}
	sendEndGameToAll(roundStartPlayers);
    } else if (numPlayersAlive()==1) {
	// Look for winner
	Player *winner=NULL;
	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		LogFile::Log("++++Player %s won", player->second->getName());
		winner = player->second;
		break;
	    }
	}
	assert(winner!=NULL);

	// Send notification to all players
	sendEndGameToAll(winner);
    } else {
	// Draw 
	vector<Player *> winners;
	LogFile::Log("Draw to exceeding number of rounds");
	for (player = players.begin(); player != players.end(); player++) {
	    if (!player->second->isStateSet(CS_DEAD)) {
		LogFile::Log("++++Player %s draw", player->second->getName());
		winners.push_back(player->second);
	    }
	}

	// Send notification
	sendEndGameToAll(winners);
    }

}

void
Game::Reset()
{
    // Remove all non players and disconnected players from creature
    // list
    vector<BaseCreature *> newCreatureList;
    vector<BaseCreature *>::iterator creature;
    for (creature=creatures.begin(); creature!=creatures.end(); creature++) {
	Player *thePlayer = dynamic_cast<Player *>(*creature);
	if (thePlayer!=NULL && !thePlayer->connection().isDisabled()) {
	    newCreatureList.push_back(*creature);
	} else {
	    // delete monsters, disconnected players are deleted later
	    if (thePlayer==NULL) delete *creature;
	}
    }
    creatures = newCreatureList;

    // Reset all of the players
    map<int, Player *>::iterator player;
    map<int, Player *> remainingPlayers;
    for (player=players.begin(); player!=players.end(); player++) {
	if (player->second->connection().isDisabled()) {
	    // Remove player from name list
	    string playerName = player->second->getName();
	    vector<string>::iterator playerNameIt = 
		find(playerNameList.begin(), playerNameList.end(),
		     playerName);
	    assert(playerNameIt!=playerNameList.end());
	    playerNameList.erase(playerNameIt);
	    delete player->second;
	} else {
	    // Reset the player settings
	    player->second->Reset();
	    remainingPlayers[player->first] = player->second;
	}
    }
    players = remainingPlayers;

    monsterNameIndex = monsterNames.begin();
}

void
Game::WaitForResponses(int Timeout, bool RemoveClient)
{
    struct timeval timeout;
    timeout.tv_sec = Timeout;
    timeout.tv_usec = 0;

    fd_set readFs, writeFs;
    int maxFD;
    bool waitingForPlayer;
    int numFDReady;
    map<int, Player *>::iterator player;
    map<int, Player *>::iterator observer;
    while (1) {
	
	waitingForPlayer = false;
	maxFD = 0;

	FD_ZERO(&readFs);
	FD_ZERO(&writeFs);

	// Accept observers 
	FD_SET(observerServerFD, &readFs);
	maxFD = max(maxFD, observerServerFD);

	// Listen for connected players
	for (player=players.begin(); player!=players.end(); player++) {
	    if (!player->second->connection().isDisabled()) {
		FD_SET(player->first, &readFs);
		maxFD = max(maxFD, player->first);
	    }

	    if (player->second->connection().pendingDataToWrite()) {
		// Need to send data to player
		FD_SET(player->first, &writeFs);
		waitingForPlayer = true;
		maxFD = max(maxFD, player->first);
	    }

	    if (player->second->amWaitingForResponse()) {
		waitingForPlayer = true;
	    } 
	}

	for (observer=observers.begin(); observer!=observers.end();
	     observer++) {
	    if (!observer->second->connection().isDisabled()) {
		FD_SET(observer->first, &readFs);
		maxFD = max(maxFD, observer->first);
	    }

	    if (observer->second->connection().pendingDataToWrite()) {
		FD_SET(observer->first, &writeFs);
		maxFD = max(maxFD, observer->first);
	    }
	}

	if (!waitingForPlayer) {
	    // All players have responded 
	    break;
	}

	numFDReady = select(maxFD+1, &readFs, &writeFs, NULL, &timeout);

	if (numFDReady==0) {
	    LogFile::Log(LogFile::DEBUG1, "Timeout for responses");
	    break;
	} else if (numFDReady<0) {
	    perror("Select in WaitForResponses failed");
	    abort();
	}

	// Observer joining
	if (FD_ISSET(observerServerFD, &readFs)) {
	    // New observer joining
	    struct sockaddr_in sa;
	    socklen_t salen;
	    int observerFD;
	    salen = sizeof(sa);
	    observerFD = 
		accept(observerServerFD, (struct sockaddr *)&sa, &salen);
	    
	    if (observerFD<0) {
		perror("Accept failed for observer");
		LogFile::Log("Observer connection failed");
	    }

	    // Set connection non blocking
	    int flags;
	    flags = fcntl(observerFD, F_GETFL);
	    if (flags<0) {
		perror("fcntl F_GETFL failed");
		exit(1);
	    }
	    flags = flags | O_NONBLOCK;
	    if (fcntl(observerFD, F_SETFL, flags)<0) {
		perror("failed to make connection non blocking");
		exit(1);
	    }

	    Player *newObserver = new Player(observerFD, 
					       SPELLCAST_PROTOCOL_VERSION,
					       playerNameList, true);
	    addObserver(observerFD, newObserver);

	    // Send initial information
	    // Send new player info
	    
	    newObserver->sendStartGame(turnTimeout, maxRoundCount);
	    map<int, Player *>::iterator newPlayer;
	    for (newPlayer = players.begin(); 
		 newPlayer != players.end(); 
		 newPlayer++) {
		newObserver->sendNewPlayerInfo(newPlayer->second);
	    }

	    vector<BaseCreature *>::iterator creature;
	    for (creature = creatures.begin();
		 creature != creatures.end();
		 creature++) {
		newObserver->sendCreatureState(*creature);
	    }
	    newObserver->sendCreatureStateEnd();
	}


	// Check for player communication
	for (player=players.begin(); player!=players.end(); player++) {
	    if (FD_ISSET(player->first, &readFs)) {
		try {
		    // Data to read
		    player->second->connection().readData();
		} catch (RemoteConnection::ConnectionError Error) {
		    PlayerDisconnected(Error, RemoveClient);
		}

		// Process messages if any received
		while (player->second->connection().haveMessageToRead()) {
		    player->second->ProcessMessage();
		}
	    }

	    if (FD_ISSET(player->first, &writeFs)) {
		try {
		    // Can write more data to player
		    player->second->connection().writeData();
		} catch (RemoteConnection::ConnectionError Error) {
		    PlayerDisconnected(Error, RemoveClient);
		}
	    }
	}

	for (observer=observers.begin(); observer!=observers.end(); 
	     observer++) {
	    try {
		if (FD_ISSET(observer->first, &readFs)) {
		    observer->second->connection().readData();
		}
		
		// Process messages if any received
		while (observer->second->connection().haveMessageToRead()){
		    observer->second->ProcessMessage();
		}

		if (FD_ISSET(observer->first, &writeFs)) {
		    observer->second->connection().writeData();

		}
	    } catch (RemoteConnection::ConnectionError Error) {
		ObserverDisconnected(Error);
	    }
	}
    }
}

NonWizard *
Game::createMonster(Spell SummonSpell, Wizard *Owner)
{
    NonWizard *monster;
    static int monsterNum = 0;

    switch (SummonSpell) {
    case SPL_SUMMON_GOBLIN:
	monster = new NonWizard(SC_MONSTER_GOBLIN, 1, 1, Owner);
	break;

    case SPL_SUMMON_OGRE:
	monster = new NonWizard(SC_MONSTER_OGRE, 2, 2, Owner);
	break;

    case SPL_SUMMON_TROLL:
	monster = new NonWizard(SC_MONSTER_TROLL, 3, 3, Owner);
	break;

    case SPL_SUMMON_GIANT:
	monster = new NonWizard(SC_MONSTER_GIANT, 4, 4, Owner);
	break;

    case SPL_SUMMON_FIRE_ELEMENTAL:
	monster = new Elemental(Owner, SC_MONSTER_FIRE_ELEMENTAL);
	break;

    case SPL_SUMMON_ICE_ELEMENTAL:
	monster = new Elemental(Owner, SC_MONSTER_ICE_ELEMENTAL);
	break;

    default:
	printf("Not a summoning spell\n");
	assert(0);
    }

    if (monsterNameIndex==monsterNames.end()) {
	monsterNum++;
	char *monsterName;
	asprintf(&monsterName, "monster%i", monsterNum);
	monster->setName(monsterName);
	free(monsterName);
    } else {
	// Use monster name list
	monster->setName(monsterNameIndex->c_str());
	monsterNameIndex++;
    }
    
    return monster;
}

void
Game::monsterAttack(NonWizard *Source, BaseCreature *Target) 
{
    int damage;
    string monsterMsg;
    map<int, Player *>::iterator player;
    map<int, Player *>::iterator observer;
    

    damage = Target->applyMonsterAttack(&monsterMsg, Source);
    
    for (player = players.begin(); player != players.end(); player++) {
	player->second->sendMonsterAttack(Source->getID(), Target->getID(),
					  damage, monsterMsg);
    }
    for (observer = observers.begin(); observer != observers.end(); 
	 observer++) {
	observer->second->sendMonsterAttack(Source->getID(), Target->getID(),
					    damage, monsterMsg);
    }
}


void
Game::SendEventMessageToAll(int EventType, int MessageSource,
			    int Source, int Target, int Misc,
			    const string &Message)
{
    map<int, Player *>::iterator player;
    map<int, Player *>::iterator observer;

    for (player = players.begin(); player != players.end(); player++) {
	player->second->sendEventMessage(EventType, 
					 MessageSource,
					 Source,
					 Target,
					 Misc,
					 Message);
    }
    for (observer = observers.begin(); observer != observers.end(); 
	 observer++) {
	observer->second->sendEventMessage(EventType, 
					   MessageSource,
					   Source,
					   Target,
					   Misc,
					   Message);
    }
}

void
Game::SendEventMessageToAll(const EventMessage &Message) 
{
    SendEventMessageToAll(Message.eventType, 
			  Message.messageSource,
			  Message.source,
			  Message.target,
			  Message.misc,
			  Message.message);
}

void
Game::SendEventMessageToAllCache(int EventType, int MessageSource,
				 int Source, int Target, int Misc,
				 const string &Message)
{
    eventMessageCache.push_back(EventMessage(EventType,
					     MessageSource,
					     Source, Target, Misc,
					     Message));
}

void
Game::SendEventMessageToAllCacheFlush()
{
    vector<EventMessage>::iterator event;
    for (event=eventMessageCache.begin(); event!=eventMessageCache.end();
	 event++) {
	SendEventMessageToAll(*event);
    }
    eventMessageCache.clear();
}

void
Game::PlayerDisconnected(const RemoteConnection::ConnectionError &Error,
			 bool RemovePlayer)
{
    assert(players.find(Error.getFD())!=players.end());
    Player *player = players[Error.getFD()];

    if (Error.getErrorNumber()==0) {
	LogFile::Log("Player %s has disconnected", player->getName());
    } else {
	LogFile::Log("Error (%i) talking to Player %s.", 
		     Error.getFD(), player->getName());
    }

    // Make sure we're not waiting for a player response
    player->clearWaitForResponse();

    // Set player to dead so we don't ask any more questions
    player->setStateBits(CS_DEAD);

    if (RemovePlayer) {
	LogFile::Log("Removing player %s", player->getName());
	string playerName = player->getName();
	vector<string>::iterator playerNameIt = 
	    find(playerNameList.begin(), playerNameList.end(),
		 playerName);
	// It may not exist yet in the playerNameList
	if (playerNameIt!=playerNameList.end()) {
	    playerNameList.erase(playerNameIt);
	}

	players.erase(Error.getFD());
	close(Error.getFD());
	vector<BaseCreature *>::iterator creature;
	creature = find(creatures.begin(), creatures.end(), 
			dynamic_cast<BaseCreature *>(player));
	assert(creature!=creatures.end());
	creatures.erase(creature);
    }
}

void
Game::ObserverDisconnected(const RemoteConnection::ConnectionError &Error)
{
    assert(observers.find(Error.getFD())!=observers.end());
    Player *observer = observers[Error.getFD()];

    if (Error.getErrorNumber()==0) {
	LogFile::Log("Observer %s has disconnected", observer->getName());
    } else {
	LogFile::Log("Error (%i) talking to Observer %s.", 
		     Error.getFD(), observer->getName());
    }
    observers.erase(Error.getFD());

    // Remove from name list too
    string obsName = observer->getName();
    vector<string>::iterator obsNameIt = 
	find(playerNameList.begin(), playerNameList.end(),
	     obsName);
    if (obsNameIt!=playerNameList.end()) {
	playerNameList.erase(obsNameIt);
    }

}
