/**********************************************************************
   Copyright (C) Christopher Yeoh <cyeoh@samba.org> 2005
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**********************************************************************/
#include <assert.h>
#include <string.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <stdlib.h>

#include "Player.H"
#include "Message.H"
#include "protocol.h"
#include "NonWizard.H"
#include "LogFile.H"
#include "Elemental.H"

vector<Player::SpellCast> Player::SpellCastPredicateSpellEqual::dummy;

Player::Player(int FD, int ProtocolVersion, vector<string> &PlayerNameList,
	       bool Observer)
    : remoteConnection(FD), waitingForResponse(true),
      leftHandSpellTarget(false), rightHandSpellTarget(false), 
      handParalysed(0), handCharmed(0), usedShortLightning(false),
      playerNameList(PlayerNameList), 
      amObserver(Observer)
{
    // Send welcome message
    LogFile::Log(LogFile::DEBUG1, "Sending welcome message");
    const char *welcomeMessage = "Welcome to SC Server";
    int index = 0;

    Message *msg;
    msg = Message::createMessage(MSG_ASK_WELCOME, sizeof(int) 
				 + strlen(welcomeMessage)+1);

    ProtocolVersion = htonl(ProtocolVersion);
    memcpy(msg->data, &ProtocolVersion, sizeof(int));
    index += sizeof(int);
    
    strcpy(msg->data+index, welcomeMessage);

    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
    setWaitForResponse();
}

void
Player::sendStartGame(int Timeout, int MaxRounds)
{
    // Send own ID, and timeout
    int playerID = htonl(getID());
    Timeout = htonl(Timeout);
    MaxRounds = htonl(MaxRounds);
    int index = 0;

    Message *msg = Message::createMessage(MSG_SEND_START_GAME,
					  sizeof(int)*3);
    memcpy(msg->data, &playerID, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &Timeout, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &MaxRounds, sizeof(int));
    index += sizeof(int);

    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
}

void
Player::sendNewPlayerInfo(Player *NewPlayer)
{
    // Send ID, Name
    int playerID = htonl(NewPlayer->getID());
    const char *playerName = NewPlayer->getName();
    int playerNameLength = strlen(playerName)+1;

    Message *msg = Message::createMessage(MSG_SEND_NEWPLAYER_INFO,
					  sizeof(int)*2+playerNameLength);

    int index = 0;
    memcpy(msg->data, &playerID, sizeof(int));
    index += sizeof(int);
    playerNameLength = htonl(playerNameLength);
    memcpy(msg->data+index, &playerNameLength, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, playerName, strlen(playerName)+1);

    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
}

void 
Player::sendNewMonsterInfo(NonWizard *Monster)
{
    // Send ID, Type, OwnerID, Name
    int monsterID = htonl(Monster->getID());
    int monsterType = htonl(Monster->getMonsterType());
    int ownerID = htonl(Monster->getOwner()->getID());
    const char *monsterName = Monster->getName();
    int monsterNameLength = strlen(monsterName)+1;

    Message *msg = Message::createMessage(MSG_SEND_NEW_MONSTER_INFO,
					  sizeof(int)*4+monsterNameLength);
    int index = 0;
    memcpy(msg->data+index, &monsterID, sizeof(int));
    index += sizeof(int);

    memcpy(msg->data+index, &monsterType, sizeof(int));
    index += sizeof(int);

    memcpy(msg->data+index, &ownerID, sizeof(int));
    index += sizeof(int);

    monsterNameLength = htonl(monsterNameLength);
    memcpy(msg->data+index, &monsterNameLength, sizeof(int));
    index += sizeof(int);

    memcpy(msg->data+index, monsterName, strlen(monsterName)+1);

    assert(msg->Ok());
    
    remoteConnection.writeMessage(msg);
    free(msg);
}

void
Player::askForGestures()
{
    Message *msg;
    msg = Message::createMessage(MSG_ASK_FOR_GESTURES, 0);
    remoteConnection.writeMessage(msg);
    free(msg);
    setWaitForResponse();

    // Set default gestures
    addGestures(GST_NOTHING, GST_NOTHING);
}

void 
Player::sendGesturesSeen(Player *OfPlayer)
{
    Gesture leftHand = GST_FOG, rightHand = GST_FOG;
    int playerID = htonl(OfPlayer->getID());
    int targetAntiSpelled = 0;

    if (!isStateSet(CS_BLINDED|CS_PERM_BLINDED) 
	&& !OfPlayer->isStateSet(CS_INVISIBILITY|CS_PERM_INVISIBILITY)) {
	OfPlayer->getGestureHistory()
	    .getCurrentGestures(&leftHand, &rightHand);
    }
    
    if (OfPlayer==this) {
	// You always know what your own gestures are whether
	// blind or invisible
	OfPlayer->getGestureHistory()
	    .getCurrentGestures(&leftHand, &rightHand);
    }


    if (leftHand==GST_ANTISPELL) {
	// Anti spell was cast on player, we need to find
	// the last real gesture
	const deque<Gesture> &gsl = OfPlayer->getGestureHistory().getLeftGestures();
	const deque<Gesture> &gsr = OfPlayer->getGestureHistory().getRightGestures();

	unsigned int i;
	for (i=1; i<gsl.size(); i++) {
	    if (gsl[i]!=GST_ANTISPELL) break;
	}
	assert(i<gsr.size());
	assert(gsr[i]!=GST_ANTISPELL);
	
	leftHand = gsl[i];
	rightHand = gsr[i];
	targetAntiSpelled = 1;
    }

    targetAntiSpelled = htonl(targetAntiSpelled);

    Message *msg;
    msg = Message::createMessage(MSG_SEND_GESTURES_SEEN, sizeof(int)*4);
    assert(sizeof(leftHand)==sizeof(int));
    leftHand = Gesture(htonl(leftHand));
    rightHand = Gesture(htonl(rightHand));
    memcpy(msg->data, &playerID, sizeof(int));
    memcpy(msg->data+sizeof(int), &leftHand, sizeof(int));
    memcpy(msg->data + sizeof(int)*2, &rightHand, sizeof(int));
    memcpy(msg->data + sizeof(int)*3, &targetAntiSpelled, sizeof(int));
    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
}

void
Player::sendGesturesSeenEnd()
{
    Message *msg;
    msg = Message::createMessage(MSG_SEND_GESTURES_SEEN, sizeof(int)*3);
    int IDZero = 0;

    // Send all zeros
    memcpy(msg->data, &IDZero, sizeof(int));
    memcpy(msg->data+sizeof(int), &IDZero, sizeof(int));
    memcpy(msg->data+sizeof(int)*2, &IDZero, sizeof(int));

    assert(msg->Ok());
       
    remoteConnection.writeMessage(msg);
    free(msg);

}

void
Player::askForSpellSelection(const vector<Spell> &LeftHandSpells,
			     const vector<Spell> &RightHandSpells)
{
    int numLeftSpells = LeftHandSpells.size();
    int numRightSpells = RightHandSpells.size();
    bool playerSurrended = false;

    if (find(LeftHandSpells.begin(), LeftHandSpells.end(), SPL_SURRENDER)
	!= LeftHandSpells.end() ||
	find(RightHandSpells.begin(), RightHandSpells.end(), SPL_SURRENDER)
	!= RightHandSpells.end()) {
	playerSurrended = true;
    }

    if (!playerSurrended && (numLeftSpells>0 || numRightSpells>0)) {
	// Keep list of possible spells for later validation
	validLHSpells = LeftHandSpells;
	validRHSpells = RightHandSpells;

	Message *msg;
	msg = Message::createMessage(MSG_ASK_FOR_SPELLS_CAST,
				     sizeof(int) * (2 + numLeftSpells
						    + numRightSpells));
	numLeftSpells = htonl(numLeftSpells);
	numRightSpells = htonl(numRightSpells);

	int index = 0;
	memcpy(msg->data, &numLeftSpells, sizeof(int));
	index += sizeof(int);
    
	assert(sizeof(Spell)==sizeof(int));

	vector<Spell>::const_iterator spell;
	int send_spell;
	for (spell=LeftHandSpells.begin(); 
	     spell!=LeftHandSpells.end(); 
	     spell++) {
	    send_spell = htonl(*spell);
	    memcpy(msg->data + index, &send_spell, sizeof(Spell)); 
	    index += sizeof(Spell);
	}

	memcpy(msg->data+index, &numRightSpells, sizeof(int));
	index += sizeof(int);
    
	for (spell=RightHandSpells.begin(); 
	     spell!=RightHandSpells.end(); spell++) {
	    send_spell = htonl(*spell);
	    memcpy(msg->data + index, &send_spell, sizeof(Spell)); 
	    index += sizeof(Spell);
	}
    
	assert(msg->Ok());

	remoteConnection.writeMessage(msg);
	free(msg);

	leftHandSpell = SPL_NONE;
	rightHandSpell = SPL_NONE;

	setWaitForResponse();
    } else {
	// No need to ask as they have no choice
	if (playerSurrended) {
	    LogFile::Log(LogFile::DEBUG2, "Player %s surrended", getName());
	    leftHandSpell = SPL_SURRENDER;
	    rightHandSpell = SPL_NONE;
	} else {
	    if (numLeftSpells==1) {
		leftHandSpell = LeftHandSpells[0];
	    } else {
		leftHandSpell = SPL_NONE;
	    }
	    
	    if (numRightSpells==1) {
		rightHandSpell = RightHandSpells[0];
	    } else {
		rightHandSpell = SPL_NONE;
	    }
	}
    }
    LogFile::Log(LogFile::DEBUG2, "Player %s default spells are %s and %s", 
		 getName(), Spells::getSpellName(leftHandSpell), 
		 Spells::getSpellName(rightHandSpell));
}

void
Player::getSpellsCast(Spell *LeftHandSpell, int *LeftTargetID,
		      Spell *RightHandSpell, int *RightTargetID)
{
    LogFile::Log(LogFile::DEBUG2, "Player %s has cast %s and %s",
		 getName(), Spells::getSpellName(leftHandSpell),
		 Spells::getSpellName(rightHandSpell));
    // Only can use short version of lightning spell once per game
    if (leftHandSpell==SPL_LIGHTNING_BOLT1
	|| rightHandSpell==SPL_LIGHTNING_BOLT1) {
	usedShortLightning = true;
    }
    
    *LeftHandSpell = leftHandSpell;
    *LeftTargetID = leftHandSpellTargetID;
    *RightHandSpell = rightHandSpell;
    *RightTargetID = rightHandSpellTargetID;
}

void
Player::askForHandToCharm(Player *Target)
{
    Message *msg = Message::createMessage(MSG_ASK_CHARM_PERSON_CTRL_HAND,
					  sizeof(int));
    int targetID = htonl(Target->getID());
    memcpy(msg->data, &targetID, sizeof(int));
    
    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
    setWaitForResponse();

    // Set default
    Target->setHandCharmed(SC_LEFT_HAND);
    validCharmTargets.push_back(Target->getID());
}

int
Player::getCharmedHand() const 
{
    return handCharmed;
}

void
Player::askForHandToParalyze(Player *Target) 
{
    Message *msg = Message::createMessage(MSG_ASK_PARALYSIS_CTRL_HAND,
					  sizeof(int));
    int targetID = htonl(Target->getID());
    memcpy(msg->data, &targetID, sizeof(int));
    
    assert(msg->Ok());
    remoteConnection.writeMessage(msg);
    free(msg);
    setWaitForResponse();

    // Set default
    Target->setHandParalysed(SC_LEFT_HAND);
    validParalyseTargets.push_back(Target->getID());
}

int
Player::getParalysedHand() const
{
    return handParalysed;
}

void
Player::askCharmedGesture(Player *Target) 
{
    Message *msg = Message::createMessage(MSG_ASK_CHARM_PERSON_CTRL_GESTURE,
					  sizeof(int)*2);
    int targetID = htonl(Target->getID());
    memcpy(msg->data, &targetID, sizeof(int));
    
    int targetHand = htonl(Target->getHandCharmed());
    memcpy(msg->data+sizeof(int), &targetHand, sizeof(int));

    assert(msg->Ok());
    remoteConnection.writeMessage(msg);
    free(msg);
    setWaitForResponse();

    // Set default
    Target->setCharmGesture(GST_NOTHING);

    validCharmGestureTargets.push_back(Target->getID());
}

const map<NonWizard *, int> &
Player::getMonsterDirections() const 
{
    return monsterTargets;
}

void
Player::sendSpellCast(const SpellCast &SC) 
{
    int sourceID, targetID;
    int spell, spellWorked;
    int msgLength;

    sourceID = htonl(SC.source->getID());
    targetID = htonl(SC.target->getID());
    spell = htonl((int)SC.spellCast);
    spellWorked = htonl(SC.spellWorked ? 1 : 0);
    msgLength = SC.resultMessage.length()+1;
    
    Message *msg = Message::createMessage(MSG_SEND_SPELL_CAST, 
					  5*sizeof(int) + msgLength);
    int index = 0;
    memcpy(msg->data, &sourceID, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &targetID, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &spell, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &spellWorked, sizeof(int));
    index += sizeof(int);
    msgLength = htonl(msgLength);
    memcpy(msg->data+index, &msgLength, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, SC.resultMessage.c_str(), 
	   SC.resultMessage.length()+1);

    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
}

void Player::sendSpellCastEnd()
{
    int IDZero = 0;
    Message *msg = Message::createMessage(MSG_SEND_SPELL_CAST,
					  sizeof(int));
    memcpy(msg->data, &IDZero, sizeof(int));
    
    assert(msg->Ok());
    remoteConnection.writeMessage(msg);
    free(msg);
}

void
Player::sendMonsterAttack(int SourceID, int TargetID, int Damage,
			  const string &ResultMessage) 
{
    LogFile::Log(LogFile::DEBUG3, "Player::sendMonsterAttack %i %i %i", 
		 SourceID, TargetID, Damage);
    SourceID = htonl(SourceID);
    TargetID = htonl(TargetID);
    Damage = htonl(Damage);
    int msgLength = ResultMessage.length()+1;

    Message *msg = Message::createMessage(MSG_SEND_MONSTER_ATTACK_INFO,
					  sizeof(int)*4 + msgLength);
    msgLength = htonl(msgLength);

    int index = 0;
    memcpy(msg->data+index, &SourceID, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &TargetID, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &Damage, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &msgLength, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, ResultMessage.c_str(), 
	   strlen(ResultMessage.c_str())+1);

    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
}

void
Player::sendMonsterAttackEnd()
{
    int idZero = 0;

    Message *msg = Message::createMessage(MSG_SEND_MONSTER_ATTACK_INFO,
					  sizeof(int));
    memcpy(msg->data, &idZero, sizeof(int));
    
    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
}



void
Player::sendCreatureState(BaseCreature *Creature) 
{
    int creatureID = htonl(Creature->getID());
    int hp = htonl(Creature->getNumHitPoints());
    unsigned int sendState = htonl(Creature->getState());
    int protfromevil = htonl(Creature->getcounterProtFromEvil());
    int disease = htonl(Creature->getcounterDisease());
    int poison = htonl(Creature->getcounterPoison());
    int blindness = htonl(0);
    int invisibility = htonl(0);

    if (Wizard *w = dynamic_cast<Wizard *>(Creature)) {
      blindness = htonl(w->getcounterBlindness());
      invisibility = htonl(w->getcounterInvisibility());
    }
    
    Message *msg = Message::createMessage(MSG_SEND_CREATURE_STATE, 
					  sizeof(int)*8);

    int index = 0;
    memcpy(msg->data, &creatureID, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &hp, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &sendState, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &protfromevil, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &disease, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &poison, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &blindness, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &invisibility, sizeof(int));

    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
}

void
Player::sendCreatureStateEnd()
{
    int IDZero = 0;
    Message *msg = Message::createMessage(MSG_SEND_CREATURE_STATE,
					  sizeof(int));
    memcpy(msg->data, &IDZero, sizeof(int));
    
    assert(msg->Ok());
    
    remoteConnection.writeMessage(msg);
    free(msg);
}

void
Player::sendEndGame(Player *Winner) 
{
    vector<Player *> winner;
    winner.push_back(Winner);
    sendEndGame(winner);
}

void
Player::sendEndGame(const vector<Player *> &Winners)
{
    int numWinners = Winners.size();
    int winnerID;
    int index = 0;
    vector<Player *>::const_iterator winner;

    Message *msg = Message::createMessage(MSG_SEND_END_GAME, 
					  sizeof(int) * (numWinners+1) );
    numWinners = htonl(numWinners);
    
    memcpy(msg->data, &numWinners, sizeof(int));
    index += sizeof(int);
    
    for (winner=Winners.begin(); winner!=Winners.end(); winner++) {
	winnerID = htonl((*winner)->getID());
	memcpy(msg->data+index, &winnerID, sizeof(int));
	index += sizeof(int);
    }
    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
}

void
Player::sendRoundStart(int Round)
{
    Round = htonl(Round);
    Message *msg = Message::createMessage(MSG_SEND_ROUND_BEGIN, sizeof(int)); 
    memcpy(msg->data, &Round, sizeof(int));
    assert(msg->Ok());
    remoteConnection.writeMessage(msg);
    free(msg);
}

void
Player::sendEventMessage(int EventType, int MessageSource,
			 int Source, int Target, int Misc,
			 const string &MessageString)
{
    int messageLength = MessageString.length()+1;
    Message *msg = Message::createMessage(MSG_SEND_EVENT_INFO,
					  sizeof(int)*6+messageLength);
    int index = 0;

    EventType = htonl(EventType);
    MessageSource = htonl(MessageSource);
    Source = htonl(Source);
    Target = htonl(Target);
    Misc = htonl(Misc);
    messageLength = htonl(messageLength);

    memcpy(msg->data, &EventType, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &MessageSource, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &Source, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &Target, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &Misc, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, &messageLength, sizeof(int));
    index += sizeof(int);
    memcpy(msg->data+index, MessageString.c_str(), MessageString.length()+1);

    assert(msg->Ok());

    remoteConnection.writeMessage(msg);
    free(msg);
}

void
Player::repeatLastGestures()
{
    const GestureHistory &gh = getGestureHistory();
    const deque<Gesture> &lgs = gh.getLeftGestures();
    const deque<Gesture> &rgs = gh.getRightGestures();

    // Look for the last real gestures
    unsigned int i;
    for (i=0; i<lgs.size(); i++) {
	if (lgs[i]!=GST_ANTISPELL) break;
    }

    assert(i<lgs.size());
    assert(lgs[i]!=GST_FOG);
    assert(rgs[i]!=GST_ANTISPELL && rgs[i]!=GST_FOG);
    assert(rgs.size()>=i);
    
    // Add these gestures to the front
    addGestures(lgs[i], rgs[i]);
}

void
Player::changeGestureForParalysis()
{
    const GestureHistory &gh = getGestureHistory();
    assert(handParalysed==SC_LEFT_HAND||handParalysed==SC_RIGHT_HAND);

    const deque<Gesture> gs = handParalysed == SC_LEFT_HAND 
	? gh.getLeftGestures() : gh.getRightGestures();

    // Look for the last real gestures (not the one just added though)
    unsigned int i;
    for (i=1; i<gs.size(); i++) {
	if (gs[i]!=GST_ANTISPELL) break;
    }

    assert(i<gs.size());

    if (handParalysed==SC_LEFT_HAND) {
	setLeftGesture(Spells::mapGestureForParalysis(gs[i])); 
    } else {
	setRightGesture(Spells::mapGestureForParalysis(gs[i]));
    }
}

bool
Player::applyCharmPersonSpell(string *Message, BaseCreature *Source)
{
    if (Wizard::applyCharmPersonSpell(Message, Source)) {
	Player *caster = dynamic_cast<Player *>(Source);
	assert(caster!=NULL);
	caster->askForHandToCharm(this);
	return true;
    } else {
	return false;
    }
}

void
Player::askForSpellDirections(const vector<int> &ValidTargetIDs,
			      const vector<int> &AllTargetIDs)
{
    validTargetIDs = ValidTargetIDs;
    allTargetIDs = AllTargetIDs;

    if (isStateSet(CS_BLINDED)) {
	// We can only cast on ourselves if we are blind
	validTargetIDs.clear();
	validTargetIDs.push_back(getID());
    } else if (isStateSet(CS_INVISIBILITY)) {
	// We can always target ourselves, even if invisible
	validTargetIDs.push_back(getID());
    }


    // Target self by default
    leftHandSpellTargetID = getID();
    rightHandSpellTargetID = getID();

    // By default don't need target information for spell
    rightHandSpellTarget = false;
    leftHandSpellTarget = false;

    // Certain spells always target the spellcaster or do not require
    // a target
    switch (leftHandSpell) {
    case SPL_NONE:
    case SPL_PERMANENCY:
    case SPL_DELAYED_EFFECT:
    case SPL_SURRENDER:
    case SPL_FIRESTORM:
    case SPL_ICESTORM:
	break;

    default:
	leftHandSpellTarget = true;
    }

    switch (rightHandSpell) {
    case SPL_NONE:
    case SPL_PERMANENCY:
    case SPL_DELAYED_EFFECT:
    case SPL_SURRENDER:
    case SPL_FIRESTORM:
    case SPL_ICESTORM:
	break;

    default:
	rightHandSpellTarget = true;
    }

    int LHSpell = htonl(leftHandSpell);
    int RHSpell = htonl(rightHandSpell);

    // If the target is already decided (eg self), don't ask player
    // for who to target
    if (!leftHandSpellTarget) LHSpell = htonl(SPL_NONE);
    if (!rightHandSpellTarget) RHSpell = htonl(SPL_NONE);
    
    if (leftHandSpellTarget || rightHandSpellTarget) {
	LogFile::Log(LogFile::DEBUG2, "Have to ask for spell direction");
	Message *msg = Message::createMessage(MSG_ASK_SPELL_DIRECTIONS,
					      sizeof(int) * 
					      (3 + validTargetIDs.size()));
	int index = 0;
	assert(sizeof(leftHandSpell)==sizeof(int));
	memcpy(msg->data, &LHSpell, sizeof(int));
	index += sizeof(int);
	memcpy(msg->data+index, &RHSpell, sizeof(int));
	index += sizeof(int);

	int numTargets = htonl(validTargetIDs.size());
	
	memcpy(msg->data+index, &numTargets, sizeof(int));
	index += sizeof(int);

	vector<int>::const_iterator targetID;
	int target;
	
	for (targetID=validTargetIDs.begin(); 
	     targetID!=validTargetIDs.end();
	     targetID++) {
	    target = htonl(*targetID);
	    memcpy(msg->data+index, &target, sizeof(int));
	    index += sizeof(int);
	}

	assert(msg->Ok());
	remoteConnection.writeMessage(msg);
	free(msg);

	setWaitForResponse();
    }
}

void
Player::askForMonsterDirections(const vector<int> &ValidTargetIDs)
{
    vector<NonWizard *> monsters = getMonsters();
    vector<NonWizard *> monstersToDirect;
    vector<NonWizard *>::iterator monster;
    map<NonWizard *, int> lastMonsterTargets = monsterTargets;

    monsterTargets.clear();
    validTargetIDs = ValidTargetIDs;

    for (monster = monsters.begin(); monster!=monsters.end(); monster++) {
	// Only look at live monsters
	if (!(*monster)->isStateSet(CS_DEAD)) {

	    if ((*monster)->isStateSet(CS_AMNESIA)) {
		if (lastMonsterTargets.find(*monster)!=lastMonsterTargets.end()) {
		    monsterTargets[*monster] = lastMonsterTargets[*monster];
		} else {
		    // Target no one if monster was not targetted at anyone
		    monsterTargets[*monster] = 0;
		}
	    } else if ((*monster)->isStateSet(CS_CONFUSION)) {
		// attack randomly
		int targetIndex = random() % validTargetIDs.size();
		monsterTargets[*monster] = ValidTargetIDs[targetIndex];
		LogFile::Log("Confused Monster %s will attack %s\n",
			     (*monster)->getName(), 
			     BaseCreature::mapIDToPointer(monsterTargets[*monster])->getName());
	    } else if ((*monster)->isStateSet(CS_PARALYSIS)) {
		// Don't do anything as paralysed
	    } else if (dynamic_cast<Elemental *>(*monster)==NULL) {
		// Target owner by default
		monsterTargets[*monster] = getID();
		monstersToDirect.push_back(*monster);
	    } else {
		// Do not need to ask directions for elemental monsters
		// as they attack everyone
		monsterTargets[*monster] = getID();
	    }
	}
    }
    
    if (monstersToDirect.size()>0) {
	// If we have at least one monster to direct
	Message *msg;
	msg = Message::createMessage(MSG_ASK_MONSTER_DIRECTIONS,
				     sizeof(int)*(monstersToDirect.size()+2
						  +ValidTargetIDs.size()));
	
	int numMonsters = htonl(monstersToDirect.size());

	int index = 0;
	memcpy(msg->data+index, &numMonsters, sizeof(numMonsters));
	index += sizeof(numMonsters);

	vector<NonWizard *>::iterator monSend;
	int monsterID;
	for (monSend = monstersToDirect.begin();
	     monSend != monstersToDirect.end();
	     monSend++) {
	    if (dynamic_cast<Elemental *>(*monSend)==NULL) {
		// Make no direction requests for elementals
		monsterID = htonl((*monSend)->getID());
		memcpy(msg->data+index, &monsterID, sizeof(monsterID));
		index += sizeof(monsterID);
	    }
	}

	int numTargets = htonl(ValidTargetIDs.size());
	memcpy(msg->data+index, &numTargets, sizeof(int));
	index += sizeof(int);
	
	vector<int>::const_iterator target;
	int targetID;
	for (target = ValidTargetIDs.begin();
	     target != ValidTargetIDs.end();
	     target++) {
	    targetID = htonl(*target);
	    memcpy(msg->data+index, &targetID, sizeof(int));
	    index += sizeof(int);
	}
	       
	assert(msg->Ok());
	remoteConnection.writeMessage(msg);
	free(msg);
	setWaitForResponse();
	LogFile::Log(LogFile::DEBUG2, "Asking %i for monster directions", 
		     getID());
    } else {
	LogFile::Log(LogFile::DEBUG3, "Not asking %i for monster directions", 
		     getID());
    }
}

bool
Player::amWaitingForResponse() const
{
    return waitingForResponse;
}

void
Player::clearWaitForResponse()
{
    waitingForResponse = false;
}

void
Player::setWaitForResponse() 
{
    waitingForResponse = true;
}

const RemoteConnection &
Player::connection() const
{
    return remoteConnection;
}

RemoteConnection &
Player::connection()
{
    return remoteConnection;
}

void
Player::ProcessMessage() 
{
    assert(remoteConnection.haveMessageToRead());
    
    // Pull next message off the queue
    Message *msg = remoteConnection.getNextMessage();
    assert(msg->Ok());

    LogFile::Log(LogFile::DEBUG3, "Recieved message type: %i", msg->command);

    switch (msg->command) {
    case MSG_RCV_CLIENT_DETAILS:
	// Expect name of player
	if (msg->dataLength()>0) {
	    msg->data[msg->dataLength()-1] = 0;
	    LogFile::Log("Client with name %s connected", msg->data);
	    string clientName(msg->data);
	    if (find(playerNameList.begin(), playerNameList.end(),
		     clientName)!=playerNameList.end()) {
		// Have a duplicate name and should disconnect
		LogFile::Log("Duplicate username %s rejected",
			     msg->data);
		Message *sendmsg = Message::createMessage(MSG_USERNAME_IN_USE_ALREADY, 0);
		assert(sendmsg->Ok());
		remoteConnection.writeMessage(sendmsg);
		free(sendmsg);

		DisconnectClient();
	    } else {
		playerNameList.push_back(clientName);
	    }
	    setName(msg->data);
	    clearWaitForResponse();
	} else {
	    LogFile::Log("Received dodgy MSG_RCV_CLIENT_DETAILS data");
	    // DISCONNECT CLIENT?
	}
	break;

    case MSG_RCV_GESTURES_USED:
	// Expect left hand and right hand gestures
	LogFile::Log(LogFile::DEBUG3, "Received Gestures");
	if (msg->dataLength()==sizeof(int)*2) {
	    int leftGesture, rightGesture;
	    memcpy(&leftGesture, msg->data, sizeof(int));
	    memcpy(&rightGesture, msg->data+sizeof(int), sizeof(int));
	    leftGesture = ntohl(leftGesture);
	    rightGesture = ntohl(rightGesture);

	    if (isStateSet(CS_FEAR)) {
		// Can't make GST_POINT, GST_CLAP, GST_FINGER or
		// GST_SNAP gestures when afraid
		// Just set gesture to GST_NOTHING
		switch (leftGesture) {
		case GST_POINT:
		case GST_CLAP:
		case GST_FINGER:
		case GST_SNAP:
		    leftGesture = GST_NOTHING;
		    break;
		}
		switch (rightGesture) {
		case GST_POINT:
		case GST_CLAP:
		case GST_FINGER:
		case GST_SNAP:
		    rightGesture = GST_NOTHING;
		    break;
		}
	    }

	    // Can't stab with both hands
	    if (leftGesture==GST_KNIFE && rightGesture==GST_KNIFE) {
		rightGesture = GST_NOTHING;
	    }

	    LogFile::Log(LogFile::DEBUG3, "Received gestures %s %s",
			 Spells::getGestureName((Gesture)leftGesture),
			 Spells::getGestureName((Gesture)rightGesture));

	    if (leftGesture>=0 && leftGesture<=GST_CLAP &&
		rightGesture>=0 && rightGesture<=GST_CLAP) {
		setLeftGesture((Gesture)leftGesture);
		setRightGesture((Gesture)rightGesture);
	    } else {
		LogFile::Log("Recieved dodgy MSG_RCV_GESTURES_USED parameters from %s %i %i", getName(), leftGesture, rightGesture);
	    }

	    clearWaitForResponse();
	} else {
	    LogFile::Log("Received dodgy MSG_RCV_GESTURES_USED data");
	    // DISCONNECT CLIENT?
	    clearWaitForResponse();
	}
	break;


    case MSG_RCV_SPELLS_CAST:
	// Expect left hand spell, right hand spell
	if (msg->dataLength()==sizeof(int)*2) {
	    int ls, rs;
	    Spell leftSpell, rightSpell;
	    memcpy(&ls, msg->data, sizeof(int));
	    memcpy(&rs, msg->data + sizeof(int), sizeof(int));
	    leftSpell = (Spell)ntohl(ls);
	    rightSpell = (Spell)ntohl(rs);

	    LogFile::Log(LogFile::DEBUG2, "RSC: %s Received %i %i", 
			 getName(), leftSpell, rightSpell);

	    // Check spell is from list originally given to client
	    if (leftSpell!=SPL_NONE &&
		find(validLHSpells.begin(), validLHSpells.end(), leftSpell)
		== validLHSpells.end()) {
		LogFile::Log("Client (%s) tried to send invalid spell",
			     getName());
		leftSpell = SPL_NONE;
		// DISCONNECT CLIENT?
	    }
	    if (rightSpell!=SPL_NONE &&
		find(validRHSpells.begin(), validRHSpells.end(), rightSpell)
		== validRHSpells.end()) {
		LogFile::Log("Client (%s) tried to send invalid spell",
			     getName());
		rightSpell = SPL_NONE;
		// DISCONNECT CLIENT?
	    }

	    if (Spells::spellRequiresTwoHands(leftSpell)) {
		LogFile::Log(LogFile::DEBUG1, "Spell %s cancelled because spell %s requires two hands", 
			     Spells::getSpellName(rightSpell),
			     Spells::getSpellName(leftSpell));
		rightSpell = SPL_NONE;
	    } 

	    if (Spells::spellRequiresTwoHands(rightSpell)) {
		LogFile::Log(LogFile::DEBUG1, "Spell %s cancelled because spell %s requires two hands", 
			     Spells::getSpellName(leftSpell),
			     Spells::getSpellName(rightSpell));
		leftSpell = SPL_NONE;
	    } 

	    leftHandSpell = leftSpell;
	    rightHandSpell = rightSpell;
	    clearWaitForResponse();
	}
	break;

    case MSG_RCV_SPELL_DIRECTIONS:
	// Expect left hand target, right hand target
	assert(amObserver==false);
	if (msg->dataLength()==sizeof(int)*2) {
	    // Only copy target information over if there
	    // is a choice who is targetted with spell/action
	    int tmpID;
	    if (leftHandSpellTarget) {
		memcpy(&tmpID, msg->data, sizeof(int));
		tmpID = ntohl(tmpID);
		if (tmpID && 
		    find(validTargetIDs.begin(), validTargetIDs.end(), tmpID)
		    != validTargetIDs.end()) {
		    leftHandSpellTargetID = tmpID;
		} else if (tmpID==0) {
		    leftHandSpell = SPL_NONE;
		} else {
		    if (leftHandSpell==SPL_RAISE_DEAD
			&& find(allTargetIDs.begin(), allTargetIDs.end(),
				tmpID)!= allTargetIDs.end()) {
			leftHandSpellTargetID = tmpID;
		    } else {
			LogFile::Log("Client (%s) tried to target an invalid target",
				     getName());
			leftHandSpell = SPL_NONE;
		    }
		    // DISCONNECT CLIENT?
		}
	    }
	    if (rightHandSpellTarget) {
		memcpy(&tmpID, msg->data+sizeof(int),
		       sizeof(int));
		tmpID = ntohl(tmpID);
		if (tmpID && 
		    find(validTargetIDs.begin(), validTargetIDs.end(), tmpID)
		    != validTargetIDs.end()) {
		    rightHandSpellTargetID = tmpID;
		} else if (tmpID==0) {
		    rightHandSpell = SPL_NONE;
		} else {
		    if (rightHandSpell==SPL_RAISE_DEAD
			&& find(allTargetIDs.begin(), allTargetIDs.end(),
				tmpID)!= allTargetIDs.end()) {
			rightHandSpellTargetID = tmpID;
		    } else {
			LogFile::Log("Client (%s) tried to target an invalid target",
				     getName());
			rightHandSpell = SPL_NONE;
			// DISCONNECT CLIENT?
		    }
		}
	    }
	    clearWaitForResponse();

	} else {
	    clearWaitForResponse();
	    ; // DISCONNECT CLIENT?
	}
	break;

    case MSG_RCV_MONSTER_DIRECTIONS:
	// Receives all monster directions at once
	if (msg->dataLength()>=sizeof(int)*2) {
	    int numMonsters;
	    int monsterID;
	    int targetID;
	    memcpy(&numMonsters, msg->data, sizeof(int));
	    numMonsters = ntohl(numMonsters);

	    // Check incoming data is of correct size
	    if (numMonsters*sizeof(int)*2+sizeof(int) == msg->dataLength()) {

		int index = sizeof(int);
		for (int i=0; i<numMonsters; i++) {
		    memcpy(&monsterID, msg->data+index, sizeof(int));
		    index += sizeof(int);
		    monsterID = ntohl(monsterID);
		    
		    memcpy(&targetID, msg->data+index, sizeof(int));
		    index += sizeof(int);
		    targetID = ntohl(targetID);
		    
		    LogFile::Log(LogFile::DEBUG2, 
				 "Received monster directions: %i %i",
				 monsterID, targetID);
		    NonWizard *monster = NULL;
		    if (BaseCreature::validID(monsterID)) {
			monster = dynamic_cast<NonWizard *>(BaseCreature::mapIDToPointer(monsterID));
		    }

		    // Check we have a monster
		    if (monster==NULL) {
			LogFile::Log("Player %s attempted to direct monster which does not exist", getName());
			// DISCONNECT CLIENT?
		    } else {
			// Check monster is owned by this player
			const vector<NonWizard *> &playerMonsters = getMonsters();
			if (find(playerMonsters.begin(), playerMonsters.end(),
				 monster)==playerMonsters.end()) {
			    LogFile::Log("Player %s attempted to direct monster which they do not own", getName());
			    // Player does not own this monster
			    // DISCONNECT CLIENT?
			} else {
			    if (monsterTargets.find(monster)==monsterTargets.end())
				{
				    // Possibly an elemental
				    // But player should not be directing them
				    // DISCONNECT CLIENT?
				    LogFile::Log("Player %s attempted to direct monster which they should not (elemental?)", getName());
				} else {
				    if (find(validTargetIDs.begin(),
					     validTargetIDs.end(), targetID) 
					!= validTargetIDs.end()) {
					monsterTargets[monster] = targetID;
				    } else {
					// Target is not valid
					LogFile::Log("Client (%s) tried to target an invalid target with a monster", getName());
				    }
				}
			}
		    }
		}
	    } else {
		LogFile::Log("Client (%s) send invalid number of monster targets", getName());
	    }
	    clearWaitForResponse();

	} else {
	    clearWaitForResponse();
	    // DISCONNECT CLIENT?
	}
	break;
    case MSG_RCV_CHARM_PERSON_CTRL_HAND:
	// Expect id, hand
	if (msg->dataLength()==sizeof(int)*2) {
	    int playerID;
	    int hand;
	    memcpy(&playerID, msg->data, sizeof(int));
	    memcpy(&hand, msg->data+sizeof(int), sizeof(int));
	    playerID = ntohl(playerID);
	    hand = ntohl(hand);
	    
	    Player *target = NULL;

	    if (find(validCharmTargets.begin(), validCharmTargets.end(),
		     playerID)!=validCharmTargets.end()) {
		target = dynamic_cast<Player *>(mapIDToPointer(playerID));
	    }
	    if (target && (hand==SC_LEFT_HAND || hand==SC_RIGHT_HAND) ) {
		target->setHandCharmed(hand);
		clearWaitForResponse();
	    } else {
		LogFile::Log("Client (%s) sent invalid MSG_RCV_CHARM_PERSON_CTRL_HAND",
			     getName());
		// DISCONNECT CLIENT?
		clearWaitForResponse();
	    }
	} else {
	    LogFile::Log("Client (%s) sent invalid MSG_RCV_CHARM_PERSON_CTRL_HAND",
			 getName());
	    // DISCONNECT CLIENT?
	    clearWaitForResponse();
	}
	break;

    case MSG_RCV_PARALYSIS_CTRL_HAND:
	// Expect id, hand
	if (msg->dataLength()==sizeof(int)*2) {
	    int playerID;
	    int hand;
	    memcpy(&playerID, msg->data, sizeof(int));
	    memcpy(&hand, msg->data+sizeof(int), sizeof(int));
	    playerID = ntohl(playerID);
	    hand = ntohl(hand);
	    
	    Player *target = NULL;
	    if (find(validParalyseTargets.begin(), validParalyseTargets.end(),
		     playerID)!=validParalyseTargets.end()) {
		target = dynamic_cast<Player *>(mapIDToPointer(playerID));
	    }
	    if (target && (hand==SC_LEFT_HAND || hand==SC_RIGHT_HAND) ) {
		target->setHandParalysed(hand);
		clearWaitForResponse();
	    } else {
		LogFile::Log("Client (%s) sent invalid MSG_RCV_PARALYSIS_CTRL_HAND",
			     getName());
		// DISCONNECT CLIENT?
		clearWaitForResponse();
	    }
	} else {
	    LogFile::Log("Client (%s) sent invalid MSG_RCV_PARALYSIS_CTRL_HAND",
			 getName());
	    // DISCONNECT CLIENT?
	    clearWaitForResponse();
	}
	break;

    case MSG_RCV_CHARM_PERSON_CTRL_GESTURE:
	// Expect id, gesture
	if (msg->dataLength()==sizeof(int)*2) {
	    int playerID;
	    int gesture;
	    memcpy(&playerID, msg->data, sizeof(int));
	    memcpy(&gesture, msg->data+sizeof(int), sizeof(int));
	    playerID = ntohl(playerID);
	    gesture = ntohl(gesture);
	    
	    Player *target = NULL;
	    if (find(validCharmGestureTargets.begin(), 
		     validCharmGestureTargets.end(),
		     playerID)!=validCharmGestureTargets.end()) {
		target = dynamic_cast<Player *>(mapIDToPointer(playerID));
	    }
	    if (target && gesture>=0 && gesture< (int)GST_ANTISPELL) {
		target->setCharmGesture((Gesture)gesture);
		clearWaitForResponse();
	    } else {
		LogFile::Log("Client (%s) sent invalid MSG_RCV_CHARM_PERSON_CTRL_GESTURE",
			     getName());
		// DISCONNECT CLIENT?
		clearWaitForResponse();
	    }
	} else {
	    LogFile::Log("Client (%s) sent invalid MSG_RCV_CHARM_PERSON_CTRL_GESTURE",
			 getName());
	    // DISCONNECT CLIENT?
	    clearWaitForResponse();
	}
	break;


    default:
	LogFile::Log("Received unknown message %i. Ignoring", msg->command);
    }

    // free message
}

void
Player::Reset()
{
    waitingForResponse = false;
    usedShortLightning = false;

    validCharmTargets.clear();
    validParalyseTargets.clear();
    validCharmGestureTargets.clear();

    // Call base class method
    Wizard::Reset();
}

void
Player::setHandParalysed(int Hand)
{
    assert(Hand==SC_LEFT_HAND||Hand==SC_RIGHT_HAND);
    handParalysed = Hand;
}

int
Player::getHandParalysed() const
{
    return handParalysed;
}

void
Player::setHandCharmed(int Hand)
{
    assert(Hand==SC_LEFT_HAND||Hand==SC_RIGHT_HAND);
    handCharmed = Hand;
}

int 
Player::getHandCharmed() const
{
    return handCharmed;
}

void
Player::setCharmGesture(Gesture NewGesture)
{
    assert(handCharmed==SC_LEFT_HAND||handCharmed==SC_RIGHT_HAND);
    if (handCharmed==SC_LEFT_HAND) {
	setLeftGesture(NewGesture);
    } else {
	setRightGesture(NewGesture);
    }
}

bool
Player::haveUsedShortLightning() const
{
    return usedShortLightning;
}

void
Player::DisconnectClient() 
{
    remoteConnection.Disable();
    throw RemoteConnection::ConnectionError(remoteConnection.getFD(),
					    &remoteConnection, 
					    0);
}
