/*
 * A strange little application which is designed to partially
 * dirty pages, then allow the VM to write them back and 
 * evict them, then read the data back and make sure it is
 * correct.
 *
 * To find a bug which I seem to have authored.
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>

/*
 * Generate a unique pattern, based on the offset into
 * the file.
 */
static int pattern_for_offset(int offset)
{
	return offset|0x80808080;
}

static char *get_range(int start, int end)
{
	char *buf = malloc(end - start);
	int pattern;
	int offset_in_buf;
	int offset_in_file;
	char *p = (char *)&pattern;

	offset_in_buf = 0;
	for (offset_in_file = start, offset_in_buf = 0;
			offset_in_file < end;
			offset_in_file++, offset_in_buf++) {
		pattern = pattern_for_offset(offset_in_file & ~(sizeof(int)-1));
		buf[offset_in_buf] = p[offset_in_file & (sizeof(int)-1)];
	}
	return buf;
}

static void write_range(int fd, int start, int end)
{
	char *buf = get_range(start, end);
	int ret;

	ret = pwrite(fd, buf, end - start, start);
	if (ret != end - start) {
		perror("pwrite");
		exit(1);
	}
	free(buf);
}

static void dump(char *msg, char *buf, int start, int end)
{
	int off;
	int first = 1;

	printf("%s", msg);
	for (off = 0; off < end - start; off++) {
		if (first || (((off + start) & 15) == 0)) {
			printf("\n");
			printf("%08x:", off + start);
			if (first) {
				int i;

				for (i = 0; i < ((off + start) & 15); i++) {
					printf("   ");
					if (i == 7)
						printf(" ");
				}
			}
			first = 0;
		}
		else if (((off + start) & 7) == 0) {
			printf(" ");
		}
		printf(" %02x", ((unsigned char *)buf)[off]);
	}
	printf("\n");
}

static void check_range(int fd, int start, int end)
{
	char *buf = get_range(start, end);
	char *readbuf = malloc(end - start);
	int ret;

	ret = pread(fd, readbuf, end - start, start);
	if (ret != end - start) {
		perror("pread");
		exit(1);
	}

	if (memcmp(readbuf, buf, end - start)) {
		printf("error at offset %d, length %d\n", start, end - start);
		dump("expected", buf, start, end);
		dump("got", readbuf, start, end);
	}
	free(buf);
	free(readbuf);
}

static void check_zeroes(int fd, int start, int end)
{
	char *buf = malloc(end - start);
	int ret;
	int off;

	ret = pread(fd, buf, end - start, start);
	if (ret != end - start) {
		perror("pread (check_zeroes)");
		exit(1);
	}

	for (off = 0; off < end - start; off++) {
		if (buf[off]) {
			dump("non-zero data", buf, start, end);
			goto out;
		}
	}
out:
	free(buf);
}

#define P(n)	(n*4096)

static struct fill_struct {
	int start;
	int end;
} fill_structs[] = {
	{	P(0) + 0,		P(0) + 0x80	},

	{	P(1) - 1,		P(1) + 0x1080	},

	{	P(3) - 2,		P(3) + 0x0100	},

	{	P(4) + 0x400,		P(4) + 0x0800	},

	{	P(5) + 0,		P(5) + 0x0400	},
	{	P(5) + 0x0400,		P(5) + 0x0800	},
	{	P(5) + 0x0800,		P(5) + 0x0c00	},
	{	P(5) + 0x0c00,		P(5) + 0x1000	},

	{	P(6) + 0x0400,		P(6) + 0x0801	},

	{	P(7) + 0x0400,		P(7) + 0x07ff	},
};

void write_it(int fd)
{
	int idx;

	for (idx = 0; idx < sizeof(fill_structs)/sizeof(fill_structs[0]); idx++) {
		struct fill_struct *fs = &fill_structs[idx];

		write_range(fd, fs->start, fs->end);
	}
}

void check_it(int fd)
{
	int idx;
	int fileoff = 0;

	for (idx = 0; idx < sizeof(fill_structs)/sizeof(fill_structs[0]); idx++) {
		struct fill_struct *fs = &fill_structs[idx];

		check_zeroes(fd, fileoff, fs->start);
		check_range(fd, fs->start, fs->end);
		fileoff = fs->end;
	}
}

static void usage(void)
{
	fprintf(stderr, "Usage: vm-eviction [-c] filename\n");
	exit(1);
}

int main(int argc, char *argv[])
{
	int fd;
	char *filename;
	int check_only = 0;
	int mode;

	if (argc != 2 && argc != 3)
		usage();
	if (argc == 2) {
		filename = argv[1];
		mode = O_RDWR|O_TRUNC|O_CREAT;
	} else {
		check_only = 1;
		filename = argv[2];
		mode = O_RDONLY;
	}
	fd = open(filename, mode, 0666);
	if (fd < 0) {
		perror("create");
		exit(1);
	}
	if (!check_only) {
		write_it(fd);
		printf("Hit enter to check data\n");
		fflush(stdout);
		getchar();
	}
	check_it(fd);
	close(fd);
	exit(0);
}
