#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "tclient.h"
#include "tetris.h"
#include "specials.h"
#include "move.h"
#include "field.h"

/* the number of rows each piece takes up in each orientation */
static int blocks_height[7][4] = {
  { 1, 4, 0, 0 }, /* I */
  { 2, 0, 0, 0 }, /* O */
  { 3, 2, 3, 2 }, /* J */
  { 3, 2, 3, 2 }, /* L */
  { 3, 2, 0, 0 }, /* Z */
  { 3, 2, 0, 0 }, /* S */
  { 3, 2, 3, 2 }  /* T */
};

/* store a bitmap of one tetris row */
typedef unsigned short BROW;

/* bitmap for the sides (1100 0000 0000 0011) */
#define BWALL 0xC003

/* all possible pieces in all orientations in all columns
 * (4 rows each)
 */
static BROW blocks_bfield[7][4][TC_FIELD_WIDTH + 16][4];

/* where a block is, and what moves were taken to get there */
typedef struct move_state {
  int         orient;   /* the orientation of the block */
  int         bx;       /* the position of the block */
  int         by;
  int         move_count;
  signed char moves[MAX_MOVES]; /* moves taken to get here */
} MOVE_STATE;

/* a queue of move states */
#define MOVE_STATEQ_SIZE 100
typedef struct move_stateq {
  signed char *field;   /* field in the format out of GetField() */
  BROW        *bfield;  /* bitmapped field from field_to_bfield() */
  int         block;    /* the block in our field */
  signed char specials[MAX_SPECIALS + 1]; /* the specials we have */

  MOVE_STATE *move_state[MOVE_STATEQ_SIZE];

  int head;
  int tail;
  int count;
  int max;
} MOVE_STATEQ;

/* maximum number of move results that can be returned from find_moves() */
/* FIXME: make it dynamic (use realloc()) */
#define MAX_MOVE_RESULTS 1000

#if 1

#define HAS_SEEN(seen, orient, x, y) ((seen)[(y)][(x) + 2][(orient)])
#define SET_SEEN(seen, orient, x, y) (seen)[(y)][(x) + 2][(orient)] = 1
signed char seen[TC_FIELD_HEIGHT + 1][TC_FIELD_WIDTH + 2][4];
signed char dropped[TC_FIELD_HEIGHT + 1][TC_FIELD_WIDTH + 2][4];

#else

/* this one consumes less memory but seems slower */
#define HAS_SEEN(seen, orient, x, y) ((seen)[(y)][(orient)] & 1 << ((x) + 4))
#define SET_SEEN(seen, orient, x, y) (seen)[(y)][(orient)] |= (1 << ((x) + 4))
unsigned short seen[TC_FIELD_HEIGHT + 1][4];
unsigned short dropped[TC_FIELD_HEIGHT + 1][4];

#endif

/* place a block into 4 BROWs */
static void place_short_bfield(BROW *bfield, int block, int orient, int bx) {
  int y, x;

  for (y = 0; y < 4; y++) {
    bfield[y] = 0;
    for (x = 0; x < 4; x++) {
      if (blocks[block][orient][y][x]) {
        bfield[y] |= 1 << (TC_FIELD_WIDTH - (x + bx) + 1);
      }
    }
  }
}

/* precalculate what a block would look like if it was placed into a
 * bfield in any orientation in any column
 */
void init_beth_move() {
  int block, orient, x;

  for (block = 0; block < 7; block++) {
    for (orient = 0; orient < blockcount[block]; orient++) {
      for (x = -4; x < TC_FIELD_WIDTH + 4; x++) {
        place_short_bfield(blocks_bfield[block][orient][x + 4],
                           block, orient, x);
      }
    }
  }
}

/* Convert a field as supplied by tclient to a array of unsigned
 * shorts, one row per short.  The 12 bits in the middle are for the
 * playing space.  The 2 bits on either side are set to '0' so that
 * blockobstructed can test against BWALL separately.
 */
static void field_to_bfield(BROW *bfield, signed char *field) {
  int y, x;

  for (y = 0; y < TC_FIELD_HEIGHT; y++) {
    bfield[y] = 0;
    for (x = 0; x < TC_FIELD_WIDTH; x++) {
      if (fblock(field, x, y)) {
        bfield[y] |= 1 << (TC_FIELD_WIDTH - x + 1);
      }
    }
  }
}

static void show_bfield(BROW *bfield) {
  int y, x;
  signed char s[] = "|--............--|";

  for (y = 0; y < TC_FIELD_HEIGHT; y++) {
    printf("%2d", y);
    for (x = 0; x < 16; x++) {
      s[x + 1] = ((1 << (15 - x)) & bfield[y]) ? 'x' : ' ';
    }
    puts(s);
  }
  puts("");
}

/* given a bfield, a block, orientation, and position, return 0 if the
 * block can be placed there, 1 if it will hit something in the field,
 * or 2 if it hits the sides
 */
static int blockobstructed(BROW *bfield,
                           int block, int orient,
                           int bx, int by) {
  int y, yh;
  int side = 0;
  BROW *tb;

  /* get the height of the piece */
  yh = blocks_height[block][orient];

  /* check if the piece is above or below the field */
  if (by < 0 || by + yh > TC_FIELD_HEIGHT) {
    return 1;
  }

  /* get the piece bitmap */
  tb = blocks_bfield[block][orient][bx + 4];

  bfield += by;
  for (y = 0; y < yh; y++) {
    if (bfield[y] & tb[y]) {
      return 1;
    }
    if (tb[y] & BWALL) {
      side = 2;
    }
  }

  return side;
}

/* move a block down as far as possible, setting the entry in the
 * dropped array for each place the block could be moved to
 */
static void blockdrop(MOVE_STATEQ *move_stateq, MOVE_STATE *move_state) {
  int y, yh;
  BROW *tb;
  BROW *bfield;

  yh = blocks_height[move_stateq->block][move_state->orient];

  if (move_state->by < 0)
    return;

  bfield = move_stateq->bfield + move_state->by + 1;

  tb = blocks_bfield[move_stateq->block]
                    [move_state->orient]
                    [move_state->bx + 4];

  while (1) {
    if (move_state->by + 1 + yh > TC_FIELD_HEIGHT)
      return;
    for (y = 0; y < yh; y++)
      if (bfield[y] & tb[y])
        return;
    move_state->by++;
    bfield++;
    SET_SEEN(dropped, move_state->orient, move_state->bx, move_state->by);
  }

  return;
}

static void placeblock(signed char *field,
                       int block, int orient, int bx, int by) {
  TETRISBLOCK *tetrisblock;
  int x, y;

  tetrisblock = (TETRISBLOCK *)(blocks[block][orient]);
  for (y = 0; y < 4; y++)
    for (x = 0; x < 4; x++)
      if ((*tetrisblock)[y][x])
        fblock(field, x+bx, y+by) = (*tetrisblock)[y][x];
}

static int removelines(signed char *field, signed char *specials) {
  int x, y, o, c = 0;
  signed char *fieldptr = field;

  for (y = 0; y < TC_FIELD_HEIGHT; y++, fieldptr += TC_FIELD_WIDTH) {
    o = 0;
    /* count holes */
    for (x = 0; x < TC_FIELD_WIDTH; x++)
      if (!fieldptr[x])
        goto next;
    /* no holes */
    /* increment line count */
    c++;
    /* grab specials */
    if (specials)
      for (x = 0; x < TC_FIELD_WIDTH; x++)
        if (fieldptr[x] >= TC_SPECIAL_FIRST_SPECIAL)
          *specials++ = fieldptr[x];
    /* move field down */
    memmove(field + TC_FIELD_WIDTH, field, TC_FIELD_WIDTH * y);
    /* clear top line */
    memset(field, 0, TC_FIELD_WIDTH);
    next:
  }
  *specials++ = 0;

  return c;
}

static void add_new_specials(signed char *specials, signed char *new_specials,
                             int lines) {
  int c;
  int i, j;

  c = strlen(specials);

  for (i = 0; i < lines; i++) {
    for (j = 0; new_specials[j]; j++) {
      if (c >= MAX_SPECIALS) {
        return;
      }
      specials[c++] = new_specials[j];
      specials[c] = 0;
    }
  }
}

/* Given a field (in move_stateq) and a piece (in move_state_start),
 * drop the piece, place it in a copy of the field, and evaluate the
 * score.  The resultant field, score, and moves will be returned.
 */
static MOVE_RESULT *eval_move_state(MOVE_STATEQ *move_stateq,
                                    MOVE_STATE *move_state_start) {
  MOVE_STATE move_state;
  MOVE_RESULT *move_result;
  int lines;
  signed char new_specials[100]; /* must be at least TC_FIELD_WIDTH * 4 */

  memcpy(&move_state, move_state_start, sizeof(MOVE_STATE));
  move_result = malloc(sizeof(MOVE_RESULT));
  memcpy(move_result->field, move_stateq->field, TC_FIELD_SIZE);
  strcpy(move_result->specials, move_stateq->specials);

  blockdrop(move_stateq, &move_state);

  placeblock(move_result->field, move_stateq->block,
             move_state.orient, move_state.bx, move_state.by);
  lines = removelines(move_result->field, new_specials);
  add_new_specials(move_result->specials, new_specials, lines);

  move_result->score = eval_field(move_result->field, 1);
  move_result->score += eval_specials_in_tray(move_result->specials);
  move_result->move_count = move_state_start->move_count;
  memcpy(move_result->moves, move_state_start->moves, MAX_MOVES);

  return move_result;
}

static MOVE_STATE *new_move_state(int orient, int bx, int by) {
  MOVE_STATE *move_state;
  move_state = malloc(sizeof(MOVE_STATE));
  move_state->orient = orient;
  move_state->bx = bx;
  move_state->by = by;
  move_state->move_count = 0;
  return move_state;
}

static MOVE_STATE *dup_move_state(MOVE_STATE *old_move_state,
                                  int orient, int bx, int by,
                                  signed char move) {
  MOVE_STATE *move_state;
  move_state = malloc(sizeof(MOVE_STATE));
  move_state->orient = orient;
  move_state->bx = bx;
  move_state->by = by;
  memcpy(move_state->moves, old_move_state->moves,
         sizeof(old_move_state->moves));
  move_state->move_count = old_move_state->move_count;
  move_state->moves[move_state->move_count++] = move;
  return move_state;
}

static void push_move_state(MOVE_STATEQ *move_stateq, MOVE_STATE *move_state) {
  move_stateq->count++;
  if (move_stateq->count > move_stateq->max) {
    move_stateq->max = move_stateq->count;
  }
  move_stateq->move_state[move_stateq->tail] = move_state;
  move_stateq->tail++;
  if (move_stateq->tail == MOVE_STATEQ_SIZE) {
    move_stateq->tail = 0;
  }
}

static MOVE_STATE *shift_move_stateq(MOVE_STATEQ *move_stateq) {
  MOVE_STATE *move_state;

  if (move_stateq->count == 0) {
    return NULL;
  }

  move_stateq->count--;
  move_state = move_stateq->move_state[move_stateq->head];
  if (!move_state) {
    printf("shift_move_stateq 0!\n");
    exit(1);
  }
  move_stateq->head++;
  if (move_stateq->head == MOVE_STATEQ_SIZE) {
    move_stateq->head = 0;
  }

  return move_state;
}

/* comparison function to sort the move results from best to worst */
static int compare_move_results(const void *a, const void *b) {
  if ((*(MOVE_RESULT **)a)->score > (*(MOVE_RESULT **)b)->score) {
    return -1;
  } else {
    return (*(MOVE_RESULT **)a)->score < (*(MOVE_RESULT **)b)->score;
  }
}

/* if we get obstructed rotating while next to a wall, see if we can
 * move away from the wall
 */
static int check_rotate_wall(BROW *bfield,
                             int block, int *orient,
                             int *bx, int *by) {
  int dir = *bx < TC_FIELD_WIDTH / 2 ? 1 : -1;
  int count;

  for (count = 0; count < 2; count++) {
    int is_obstructed;

    *bx += dir;

    if (HAS_SEEN(seen, *orient, *bx, *by)) {
      return 0;
    }

    SET_SEEN(seen, *orient, *bx, *by);

    is_obstructed = blockobstructed(bfield, block, *orient, *bx, *by);

    /* if not obstructed, return "yes" */
    if (!is_obstructed) {
      return 1;
    }
  }

  return 0;
}

/* Given a starting state (current field, specials in tray, current
 * block, orientation, and position), returns NULL if the block
 * is already obstructed, otherwise returns a pointer to an array of
 * MOVE_RESULT structure describing the possible moves that can be
 * made.  The moves will be sorted from best score to worst score.
 */
MOVE_RESULT **find_moves(signed char *field, signed char *specials,
                         int block, int orient, int bx, int by) {
  MOVE_STATEQ move_stateq;
  MOVE_STATE  *move_state;
  MOVE_RESULT **move_results = malloc(MAX_MOVE_RESULTS * sizeof(MOVE_RESULT *));
  int         move_result_num = 0;

  move_stateq.field = field;
  strncpy(move_stateq.specials, specials, MAX_SPECIALS);
  move_stateq.specials[MAX_SPECIALS] = 0;

  move_stateq.bfield = malloc(sizeof(BROW) * TC_FIELD_HEIGHT);
  field_to_bfield(move_stateq.bfield, field);

  /* make sure the piece isn't already obstructed */
  if (blockobstructed(move_stateq.bfield, block, orient, bx, by)) {
    free(move_stateq.bfield);
    free(move_results);
    return NULL;
  }

  move_stateq.block = block;
  move_stateq.head = 0;
  move_stateq.tail = 0;
  move_stateq.count = 0;
  move_stateq.max = 0;

  push_move_state(&move_stateq, new_move_state(orient, bx, by));

  memset(seen, 0, sizeof(seen));
  memset(dropped, 0, sizeof(dropped));

  SET_SEEN(seen, orient, bx, by);

  while ((move_state = shift_move_stateq(&move_stateq))) {
    int move;

    if (!HAS_SEEN(dropped, move_state->orient,
                  move_state->bx, move_state->by)) {
      move_results[move_result_num++] =
          eval_move_state(&move_stateq, move_state);
      if (move_result_num >= MAX_MOVE_RESULTS) {
        printf("increase move_result_num\n");
        exit(1);
      }
    }

    /* don't play TC_MOVE_LAST_MOVE as it's TC_MOVE_DROP and we always
     * play that at the end of a sequence of moves (never in the
     * middle)
     */
    for (move = TC_MOVE_FIRST_MOVE; move < TC_MOVE_LAST_MOVE; move++) {
      int blockisobstructed;
      int bx, by, orient;

      /* if rotating, check that the block can be rotated */
      if (blockcount[block] == 1 &&
          (move == TC_ROTATE_CLOCKWISE || move == TC_ROTATE_ANTICLOCKWISE)) {
        continue;
      }

      bx = move_state->bx;
      by = move_state->by;
      orient = move_state->orient;

      /* move the block in the chosen direction */
      switch (move) {
        case TC_MOVE_LEFT:
          bx--;
          break;
        case TC_MOVE_RIGHT:
          bx++;
          break;
        case TC_ROTATE_CLOCKWISE:
          if (++orient >= blockcount[block])
            orient = 0;
          break;
        case TC_ROTATE_ANTICLOCKWISE:
          if (--orient < 0)
            orient = blockcount[block] - 1;
          break;
        case TC_MOVE_DOWN:
          by++;
          break;
      }

      /* if we've seen the block in this position before, continue */
      if (HAS_SEEN(seen, orient, bx, by)) {
        continue;
      }

      /* haven't seen the block here before; mark it */
      SET_SEEN(seen, orient, bx, by);

      /* check if the block is obstructed */
      if ((blockisobstructed =
           blockobstructed(move_stateq.bfield, block, orient, bx, by))) {

        /* if we're not rotating or if we didn't hit a wall, continue on */
        if ((move != TC_ROTATE_CLOCKWISE && move != TC_ROTATE_ANTICLOCKWISE) ||
            blockisobstructed != 2) {
          continue;
        }

        /* we are rotating and we hit a wall -- see if we can move
         * away from it
         */
        if (!check_rotate_wall(move_stateq.bfield, block, &orient, &bx, &by)) {
          continue;
        }
      }

      push_move_state(&move_stateq,
                      dup_move_state(move_state, orient, bx, by, move));
    }

    free(move_state);
  }

  free(move_stateq.bfield);

  move_results[move_result_num] = NULL;

  qsort(move_results, move_result_num, sizeof(MOVE_RESULT *),
        compare_move_results);
  return move_results;
}

/* Call find_moves() and return just the best move found.  The caller
 * will need to free the MOVE_RESULT pointer returned.
 */
MOVE_RESULT *find_best_move(signed char *field, signed char *specials,
                            int block, int orient, int bx, int by) {
  MOVE_RESULT **move_results;
  MOVE_RESULT *best_move_result;
  MOVE_RESULT **search_move_results;
  MOVE_RESULT *search_move_result;

  /* get all the possible moves */
  move_results = find_moves(field, specials, block, orient, bx, by);
  if (!move_results) {
    return NULL;
  }

  /* save the best move */
  best_move_result = move_results[0];

  /* free all the other moves and the MOVE_RESULT array */
  search_move_results = move_results + 1;
  while ((search_move_result = *search_move_results++)) {
    free(search_move_result);
  }
  free(move_results);

  return best_move_result;
}

/* Call find_best_move() to find the best move, and return the score
 * of that field.
 */
int find_best_move_score(signed char *field, signed char *specials,
                         int block, int orient, int bx, int by) {
  MOVE_RESULT *move_result;
  int score;

  move_result = find_best_move(field, specials, block, orient, bx, by);
  if (!move_result) {
    return -10000;
  }
  score = move_result->score;
  free(move_result);
  return score;
}
