/*
 *  Copyright (C) 2003 Chris Yeoh (cyeoh@samba.org)
 *  Copyright (C) 2003 Jeremy Kerr (jeremy@redfishsoftware.com.au)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <stdio.h>
#include <netdb.h>
#include <stdlib.h>
#include <strings.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>

#define _GNU_SOURCE
#include <stdio.h>

#include "tcommon.h"
#include "comm_client.h"
#include "tclient.h"
#include "tetris.h"
#include "client.h"
#include "log.h"

int ClientState = 0;
int SendField = 0;
struct game GameState;
static char *Username;

int createServerSocket(int Port)
{
	int serverSocket;
	struct sockaddr_in address;
	int sock_opt = 1;

	if ( (serverSocket = socket(PF_INET, SOCK_STREAM, 0)) < 0)
	{
		perror("Could not create socket");
		exit(1);
	}

	if (setsockopt(serverSocket, SOL_SOCKET, SO_REUSEADDR, &sock_opt, 
								 sizeof(sock_opt))!=0)
	{
		perror("setsockopt failed");
		exit(1);
	}
	
	bzero(&address, sizeof(address));
	address.sin_family = AF_INET;
	address.sin_port = htons(Port);
	address.sin_addr.s_addr = INADDR_ANY;

	if (bind(serverSocket, (struct sockaddr *)&address, sizeof(address))!=0)
	{
		perror("Failed to bind");
		exit(1);
	}

	if (listen(serverSocket, 1)!=0)
	{
		perror("Listen failed");
		exit(1);
	}

	return serverSocket;

}
		
int processConnection(int Socket)
{
	int tmp_int;
	char *teamname, *channel;

	/* protocol version check */
	readNumBytes(Socket, &tmp_int, sizeof(tmp_int));
	tmp_int = ntohl(tmp_int);
	if (tmp_int != TCOMMON_PROTOCOL_VERSION)
	{
		printf("Unexpected protocol version %i, expecting %i\n",
					 tmp_int, TCOMMON_PROTOCOL_VERSION);
		writeInt32(Socket, 1);
		writeString(Socket, "Version protocol mismatch");

		exit(1);
	}

	Username = readString(Socket);
	teamname = readString(Socket);
	channel = readString(Socket);

	printf("Received connection (%s, %s, %s)\n", Username, teamname, channel);
	
	writeInt32(Socket, 0);

	return 0;
}


int processMove(int Socket)
{
	int moveType, reset_timer = 0;
	int result = 0;

	moveType = readInt32(Socket);

	switch (moveType)
	{
	case TC_MOVE_LEFT:
		result = tetris_blockmove(-1, GameState.players[GameState.playernum-1].field);
		break;

	case TC_MOVE_RIGHT:
		result = tetris_blockmove(1, GameState.players[GameState.playernum-1].field);
		break;

	case TC_ROTATE_CLOCKWISE:
		result = tetris_blockrotate(1, GameState.players[GameState.playernum-1].field);
		break;

	case TC_ROTATE_ANTICLOCKWISE:
		result = tetris_blockrotate(-1, GameState.players[GameState.playernum-1].field);
		break;

	case TC_MOVE_DOWN:
		result = tetris_blockdown(GameState.players[GameState.playernum-1].field);
		break;

	case TC_MOVE_DROP:
		tetris_blockdrop(GameState.players[GameState.playernum-1].field);
		SendField = 1;
		reset_timer = 1;
		break;

	default:
		printf("Unknown move type\n");
		abort();
	}

	switch (moveType)
	{
	case TC_MOVE_LEFT:
	case TC_MOVE_RIGHT:
	case TC_ROTATE_CLOCKWISE:
	case TC_ROTATE_ANTICLOCKWISE:
		if (result==-1)	result = 1;
		else result = 0;
		break;

	case TC_MOVE_DROP:
		if (tetris_solidify(GameState.players[GameState.playernum-1].field)==0)
		{
			int lines_cleared, i, j;
			char specials_buffer[200];
			char *spc;

			lines_cleared = tetris_removelines(
				specials_buffer, 
				GameState.players[GameState.playernum-1].field);
			tetrinet_addsbtofield(lines_cleared, 
														GameState.players[GameState.playernum-1].field);
			i=0;
			while (specials_buffer[i]!=0)
			{
				for (j=0; j<lines_cleared; j++)
					tetris_add_special(specials_buffer[i]);
				i++;
			}
			/* send extra lines to other people */
			switch (lines_cleared)
			{
			case 4:
				spc = "cs4";
				break;
			case 3:
				spc = "cs2";
				break;
			case 2:
				spc = "cs1";
				break;
			default:
				spc = NULL;
			}
			if (spc) send_special(0, spc, GameState.playernum);
		}
		else
		{
			/* player is dead! !! */
			tbt_log(TBT_LOG_LOW, "** Player is dead\n");
			tetris_set_player_state(GameState.playernum-1, 1);
			tetris_create_dead_field(
				GameState.players[GameState.playernum-1].field);
			result = 2;
		}
		SendField = 1;
		break;

	default:
	}

	writeInt32(Socket, result);

	return reset_timer;
}


static void sendFieldNum(int Socket)
{
	int fieldNum;
	fieldNum = readInt32(Socket);

	if (fieldNum>=0 && fieldNum<6 && !tetris_get_player_state(fieldNum))
	{
		writeInt32(Socket, 0);
		writeNumBytes(Socket, GameState.players[fieldNum].field, 
									FIELD_WIDTH*FIELD_HEIGHT);
	}
	else
	{
		writeInt32(Socket, 1);
	}
}


static int mapToClientBlock(int BlockNum, int Orient)
{
	switch (BlockNum)
	{
	case -1:
		return TC_BLOCK_NONE;

	case 0:
		switch (Orient)
		{
		case 0:
			return TC_BLOCK_I_1;

		case 1:
			return TC_BLOCK_I_2;
		}
		
	case 1:
		return TC_BLOCK_O;

	case 2:
		switch (Orient)
		{
		case 0:
			return TC_BLOCK_J_4;

		case 1:
			return TC_BLOCK_J_3;
			
		case 2:
			return TC_BLOCK_J_2;

		case 3:
			return TC_BLOCK_J_1;
		}

	case 3:
		switch (Orient)
		{
		case 0:
			return TC_BLOCK_L_2;
			
		case 1:
			return TC_BLOCK_L_1;

		case 2:
			return TC_BLOCK_L_4;

		case 3:
			return TC_BLOCK_L_3;
			}

	case 4:
		switch (Orient)
		{
		case 0:
			return TC_BLOCK_Z_2;

		case 1:
			return TC_BLOCK_Z_1;
		}
		
	case 5:
		switch (Orient)
		{
		case 0:
			return TC_BLOCK_S_2;
			
		case 1:
			return TC_BLOCK_S_1;
		}

	case 6:
		switch (Orient)
		{
		case 0:
			return TC_BLOCK_T_4;

		case 1:
			return TC_BLOCK_T_3;

		case 2:
			return TC_BLOCK_T_2;

		case 3:
			return TC_BLOCK_T_1;
		}
	}
}

static void sendCurrentBlock(int Socket)
{
	int x, y, tmp;

	/* Send status */
	writeInt32(Socket, tetris_get_player_state(GameState.playernum-1));
	tmp = mapToClientBlock(get_current_block(), get_current_orient());
	writeInt32(Socket, tmp);
	get_current_block_position(&x, &y);
#ifdef DEBUG
	printf("Sending Current block: %i %i %i", tmp, x, y);
#endif
	writeInt32(Socket, x);
	writeInt32(Socket, y);
}


static void sendPendingBlock(int Socket)
{
	writeInt32(Socket, mapToClientBlock(
							 tetris_pending_block(),
							 tetris_pending_block_orient()));
}

static int useSpecial(int Socket)
{
	int player;
	char special;
	char *ss;
	int updateField = 0;

	player = readInt32(Socket);
	
	special = tetris_get_specials()[0];

		/* Check we actually have a special */
	if (special)
	{
		/* Send special to another player */
		switch (special)
		{
		case TC_SPECIAL_ADD_LINE:
			ss = "a";
			break;
				
		case TC_SPECIAL_CLEAR_SPECIALS:
			ss = "b";
			break;

		case TC_SPECIAL_CLEAR_LINE:
			ss = "c";
			break;
				
		case TC_SPECIAL_BLOCK_GRAVITY:
			ss = "g";
			break;
				
		case TC_SPECIAL_NUKE:
			ss = "n";
			break;
				
		case TC_SPECIAL_BLOCK_BOMB:
			ss = "o";
			break;

		case TC_SPECIAL_CLEAR_RANDOM:
			ss = "r";
			break;

		case TC_SPECIAL_SWITCH_FIELDS:
			ss = "s";
			break;

		case TC_SPECIAL_BLOCK_QUAKE:
			ss = "q";
			break;

		default:
			printf("** Unknown special %i ** Aborting\n", special);
			abort();
		}


		/* Still need to check that player exists! */
		if (player<0 || player>=6)
		{
			printf("Invalid player number %i\n", player);
			abort();
		}

		/* Check player is alive */
		if (tetris_get_player_state(player)==0)
		{
			tbt_log(TBT_LOG_MEDIUM, "Sending special %s from %i to %i\n",
							ss, player+1, GameState.playernum);
			tbt_log(TBT_LOG_MEDIUM, "Player state is %i %i %i %i %i %i\n",
							tetris_get_player_state(0),
							tetris_get_player_state(1),
							tetris_get_player_state(2),
							tetris_get_player_state(3),
							tetris_get_player_state(4),
							tetris_get_player_state(5));

			send_special(player+1, ss, GameState.playernum);

			if (player == GameState.playernum-1 || special==TC_SPECIAL_SWITCH_FIELDS)
			{
				updateField = 1;
				switch (special)
				{
				case TC_SPECIAL_ADD_LINE:
					if (tetris_addlines(1, GameState.players[player].field, 1))
					{
						tetris_set_player_state(GameState.playernum-1, 1);
						tetris_create_dead_field(
							GameState.players[GameState.playernum-1].field);
					}
					break;

				case TC_SPECIAL_CLEAR_LINE:
					tetris_clearline(GameState.players[player].field);
					break;

				case TC_SPECIAL_NUKE:
					memset(GameState.players[player].field, 0, FIELD_WIDTH*FIELD_HEIGHT);
					break;

				case TC_SPECIAL_CLEAR_RANDOM:
					tetris_clear_random(GameState.players[player].field);
					break;

				case TC_SPECIAL_CLEAR_SPECIALS:
					tetris_clear_specials(GameState.players[player].field);
					break;

				case TC_SPECIAL_BLOCK_GRAVITY:
					tetris_block_gravity(GameState.players[player].field);
					break;

				case TC_SPECIAL_BLOCK_QUAKE:
					tetris_block_quake(GameState.players[player].field);
					break;

				case TC_SPECIAL_BLOCK_BOMB:
					tetris_block_bomb(GameState.players[player].field);
					break;

				case TC_SPECIAL_SWITCH_FIELDS:
					tetris_switch_fields(GameState.players[GameState.playernum-1].field,
															 GameState.players[player].field);
					break;

				default:
					tbt_log(TBT_LOG_HIGH, "Special %i on self unimplemented\n", special);

				}
			}

			tetris_delete_first_special();
			writeInt32(Socket, 0);
		}
		else
		{
			/* Player is dead */
			writeInt32(Socket, 2);
		}
	}
	else
	{
		writeInt32(Socket, 1);
	}
	return updateField;
}
		
static void messageToSend(int Socket)
{
	int length;
	char *message;
	length = readInt32(Socket);
	message = malloc(sizeof(length));
	readNumBytes(Socket, message, length);
}

int processClientMessage(int Socket, int *UpdateField, int *ResetTimer)
{
	int command = 0;

	*UpdateField = 0;
	*ResetTimer = 0;

	if (readNumBytes(Socket, &command, sizeof(int))==0)
	{
		return 0;
	}

	command = ntohl(command);

#ifdef DEBUG
	printf("Received command %i\n", command);
#endif
	switch (command)
	{
	case _TC_CONNECT:
		/* Eventually want to fork and exec here to
			 be able to handle multiple clients */
		processConnection(Socket);
		break;

	case _TC_START:
		ClientState = CC_CLIENT_READY;
		tetris_set_all_players_alive();
		break;

	case _TC_COMMAND_MOVE:
		*ResetTimer = processMove(Socket);
		break;

	case _TC_COMMAND_GET_FIELD:
		sendFieldNum(Socket);
		break;

	case _TC_COMMAND_GET_CURRENT_BLOCK:
		sendCurrentBlock(Socket);
		break;

	case _TC_COMMAND_GET_NEXT_BLOCK:
		sendPendingBlock(Socket);
		break;

	case _TC_COMMAND_GET_SPECIALS:
	{
		char *specials;
		specials = tetris_get_specials();
		writeInt32(Socket, strlen(specials));
		if (strlen(specials)>0)
		{
			writeNumBytes(Socket, specials, strlen(specials));
		}
		break;
	}

	case _TC_COMMAND_USE_SPECIAL:
		*UpdateField = useSpecial(Socket);
		break;
		
	case _TC_COMMAND_DISCARD_SPECIAL:
		tetris_delete_first_special();
		break;


	case _TC_COMMAND_GET_MESSAGE:
		/* Not yet implemented */
		writeInt32(Socket, 0);
		break;

	case _TC_COMMAND_SEND_MESSAGE:
		messageToSend(Socket);
		break;

	default:
		printf("Unrecognized command from client (%i)\n", command);
		exit(1);
	}

	return 1;
}

int startClient(int Socket, struct TC_GameData *GameData)
{
	int i;

	/* Send ok status */
	writeInt32(Socket, 0);
	writeInt32(Socket, GameData->NumberOfPlayers);
	writeInt32(Socket, GameData->PlayerNumber);
	writeInt32(Socket, GameData->MaxNumSpecials);
	for (i=0; i<GameData->NumberOfPlayers; i++)
	{
		writeInt32(Socket, GameData->Players[i].PlayerNumber);
		writeString(Socket, GameData->Players[i].PlayerName);
		writeString(Socket, GameData->Players[i].TeamName);
	}
	return 0;
}

char *getUsername()
{
	return Username;
}
