/*
 *  Copyright (C) 2003 Chris Yeoh (cyeoh@samba.org)
 *  Copyright (C) 2003 Mark Pulford (mark@kyne.com.au)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <termios.h>
#include <string.h>

#include "tclient.h"
#include "btetris.h"
#include "main.h"

extern struct TC_GameData *bot_gamedata;

char *Specials = NULL;
int status = 0;

void TranslateToLocalRep(int blocktype, int *lblock, int *orientation);

void PrintField(FIELD field)
{
    int i,j;
    char display;

    dprint("\n");
    for (i=0; i<TC_FIELD_HEIGHT; i++)
    {
        for (j=0; j<TC_FIELD_WIDTH; j++)
        {
            display = field[i][j];
            display += 48;
            dprint("%c", display);
        }
        dprint("\n");
    }
}

int FieldValue(FIELD field, int *NumHoles, int *Height, int *SpecialsCount)
{
    int max_height = TC_FIELD_HEIGHT; /* which is actually a minimum */
    int lowest_height = 0;
    int num_buried_holes = 0;
    int well_count_4 = 0;
    int well_count_3 = 0;
    int well_count_5 = 0;
    int x, y;
    int got_block;
    int well_accum;
    int well_started_left = 0;
    int well_started_right = 0;
    int goodness = 0;
    int lh_col;
    int max_well_depth = 0;
    int i;
    int max_height_for_column[TC_FIELD_WIDTH];
    int jaggedness = 0;

    if (SpecialsCount)
    {
        for (i=TC_SPECIAL_FIRST_SPECIAL; i<=TC_SPECIAL_LAST_SPECIAL; i++)
        {
            SpecialsCount[i-TC_SPECIAL_LAST_SPECIAL] = 0;
        }
    }

    for (x=0; x<TC_FIELD_WIDTH; x++)
    {
        max_height_for_column[x] = TC_FIELD_HEIGHT;

        if (x==0)   well_started_left = 1;
        else if (x==TC_FIELD_WIDTH-1) well_started_right = 1;
        else
        {
            well_started_left = 0;
            well_started_right =0;

        }
        for (y=0, got_block=0, well_accum=0, lh_col=0,
                     well_started_right = 0; y<TC_FIELD_HEIGHT; y++)
        {
            /* Look for buried holes */
            if (field[y][x]) 
            {
                got_block = 1;
                if (y<max_height) max_height = y;
                if (y<max_height_for_column[x]) max_height_for_column[x] = y;
                if (!lh_col && y>lowest_height) 
                {
                    lh_col = 1;
                    lowest_height = y;
                }
                well_accum = 0;

                if (field[y][x]>=TC_SPECIAL_FIRST_SPECIAL && SpecialsCount)
                {
                    SpecialsCount[field[y][x]-TC_SPECIAL_FIRST_SPECIAL]++;
                }
            }
            else 
            {
                if (got_block) num_buried_holes++;

                /* well detection */
                if (x>0 && field[y][x-1] && !well_started_left) well_started_left = 1;
                if (x<TC_FIELD_WIDTH-1 && field[y][x+1] && !well_started_right)
                    well_started_right = 1;
                if (well_started_left && well_started_right) well_accum++;
                if (well_accum==3) well_count_3++;
                if (well_accum==4)
                {
                    well_count_3--;
                    well_count_4++;
                }
                if (well_accum==5)
                {
                    well_count_5++;
                    well_count_4--;
                }
                if (well_accum>max_well_depth) max_well_depth = well_accum;
            }
        }
    }

    max_height = TC_FIELD_HEIGHT-max_height-1;
    lowest_height = TC_FIELD_HEIGHT-lowest_height-1;
    goodness = 0;
    goodness -= num_buried_holes*250;

    /* Only more than one well is bad */
/*  if (well_count_3 + well_count_4 > 1) */
/*  { */
    goodness -= (well_count_3+well_count_4*30);
/*  } */

    /* See how jagged our field looks like */
    for (x=1; x<TC_FIELD_WIDTH; x++)
    {
        jaggedness += abs(max_height_for_column[x]-max_height_for_column[x-1]);
    }

    goodness -= jaggedness*10;

    goodness -= max_height*20;
    if (max_height>10) goodness -= (max_height-10)*(max_height-10)*10;

    /* Don't like too much unevenness */
    if (max_height-lowest_height>5)
        goodness -= (max_height - lowest_height - 5)*50;

    /* Really deep wells are bad */
    if (max_well_depth>5)
        goodness -= (max_well_depth-5)*(max_well_depth-5)*20;

/*    dprint("\n\nTest field:"); */
/*  PrintField(field); */
/*    dprint("Ht=%i, bh=%i, w3=%i, w4=%i, gd %i\n", max_height,
                 num_buried_holes, well_count_3, well_count_4, goodness); */

    if (NumHoles) *NumHoles = num_buried_holes;
    if (Height) *Height = max_height;

    return goodness;

}

int GetBestMove(int block, int xpos, int ypos, int orientation, FIELD field,
                                 int *best_position, int *best_orientation, int recurse,
                                int best_value)
{
    FIELD tmp_field;
    int rotation;
    int minx, maxx;
    int field_value;
    int lines_removed;
    int next_block;
    int next_orientation;

/* dprint("Input field\n"); */
/* PrintField(field); */
/* dprint("Given block %i, orientation %i\n", block, orientation); */

    if (recurse)
    {
        int rnb;
        GetNextBlock(&rnb);
        TranslateToLocalRep(rnb, &next_block, &next_orientation);
        dprint("Next block is %i %i %i\n", rnb, next_block, next_orientation);
    }

    for (rotation=0; rotation<num_rotations_for_blocktype(block); rotation++)
    {
        /* dprint("+++++++++++++++++++Rotation\n"); */
        get_valid_x_positions(block, orientation, &minx, &maxx);
/*      dprint("valid x positions %i %i\n", minx, maxx);  */

        for (xpos = minx; xpos<=maxx; xpos++)
        {
            /* Make a copy of the field */
            memcpy(tmp_field, field, TC_FIELD_SIZE);

            /* Set block to drop */
/*          dprint("Set current block: %i %i %i %i\n",  */
/*                       block, orientation, xpos, ypos); */
            set_current_block(block, orientation, xpos, ypos);

            /* drop the block */
            tetris_blockdrop(tmp_field);
            tetris_solidify(tmp_field);

            /* Check if line removal will happen */
            lines_removed = tetris_removelines(NULL, tmp_field);

            if (recurse)
            {
                int dummy1, dummy2;

                field_value = GetBestMove(next_block, 4, 0, next_orientation, 
                                                                    tmp_field, &dummy1, &dummy2, 0, best_value)
                    + (lines_removed*lines_removed*100);
            }
            else
            {
                field_value = FieldValue(tmp_field, NULL, NULL, NULL);
                field_value += (lines_removed)*(lines_removed)*100; 
            }

            if (field_value>best_value)
            {
                best_value = field_value;
                *best_position = xpos;
                *best_orientation = orientation;
                if (!recurse)
                {
/*                    dprint("*recursed*"); */
                }
/*              PrintField(tmp_field); */
/*                dprint("************** Found better field %i %i (value %i)\n",
                             xpos, orientation, field_value); */

            }
        }

        orientation = get_orientation_after_rotation(block, orientation, 1);

    }
    return best_value;
}


void TranslateToLocalRep(int blocktype, int *lblock, int *orientation)
{
    switch (blocktype)
    {
    case TC_BLOCK_I_1:
        *lblock = 0;
        *orientation = 0;
        return;

    case TC_BLOCK_I_2:
        *lblock = 0;
        *orientation = 1;
        return;

    case TC_BLOCK_O:
        *lblock = 1;
        *orientation = 0;
        return;

    case TC_BLOCK_J_1:
        *lblock = 2;
        *orientation = 3;
        return;

    case TC_BLOCK_J_2:
        *lblock = 2;
        *orientation = 2;
        return;

    case TC_BLOCK_J_3:
        *lblock = 2;
        *orientation = 1;
        return;

    case TC_BLOCK_J_4:
        *lblock = 2;
        *orientation = 0;
        return;

    case TC_BLOCK_L_1:
        *lblock = 3;
        *orientation = 1;
        return;

    case TC_BLOCK_L_2:
        *lblock = 3;
        *orientation = 0;
        return;

    case TC_BLOCK_L_3:
        *lblock = 3;
        *orientation = 3;
        return;

    case TC_BLOCK_L_4:
        *lblock = 3;
        *orientation = 2;
        return;

    case TC_BLOCK_Z_1:
        *lblock = 4;
        *orientation = 1;
        return;

    case TC_BLOCK_Z_2:
        *lblock = 4;
        *orientation = 0;
        return;

    case TC_BLOCK_S_1:
        *lblock = 5;
        *orientation = 1;
        return;

    case TC_BLOCK_S_2:
        *lblock = 5;
        *orientation = 0;
        return;

    case TC_BLOCK_T_1:
        *lblock = 6;
        *orientation = 3;
        return;

    case TC_BLOCK_T_2:
        *lblock = 6;
        *orientation = 2;
        return;

    case TC_BLOCK_T_3:
        *lblock = 6;
        *orientation = 1;
        return;

    case TC_BLOCK_T_4:
        *lblock = 6;
        *orientation = 0;
        return;

    default:
        dprint("oops in translation\n");
        abort();

    }

}

int HasGoodSpecial(int *Specials)
{
    return Specials[TC_SPECIAL_BLOCK_GRAVITY-TC_SPECIAL_FIRST_SPECIAL]
        || Specials[TC_SPECIAL_NUKE-TC_SPECIAL_FIRST_SPECIAL]
        || Specials[TC_SPECIAL_SWITCH_FIELDS-TC_SPECIAL_FIRST_SPECIAL];
}

void UseSpecials(FIELD field)
{
    int field_value;
    int num_holes, field_height;
    char specials[100];
    int i, try_another_special = 0, changed_field = 0;
    int num_players;
    char *fields[6];
    int field_values[6];
    int field_specials[6][9];
    int best_field_index = -1;
    int best_field_value = -20000;
    int use_special_failed = 0;

    GetSpecials(specials);

    if (specials[0])
    {
        num_players = GetNumPlayers(bot_gamedata);

        /* Get Fields for all players */
        for (i=0; i<num_players; i++)
        {
            if (GetPlayerNumber(GetPlayer(bot_gamedata,i))==GetOwnPlayerNumber(bot_gamedata))
            {
                fields[i] = NULL;
                field_values[i] = best_field_value;
            }
            else
            {
                fields[i] = GetField(GetPlayerNumber(GetPlayer(bot_gamedata,i)));
                if (fields[i])
                {
                    FIELD *tmp = (FIELD *)fields[i];
                    field_values[i] = FieldValue(*tmp, NULL, NULL,
                                                 field_specials[i]);
                }
                else
                {
                    field_values[i] = -20000;
                }
                if (field_values[i]>best_field_value) 
                {
                    best_field_index = i;
                    best_field_value = field_values[i];
                }
            }
        }

        for (i=0; specials[i]!=0; i++)
        {
            field_value = FieldValue(field, &num_holes, &field_height, NULL);
            try_another_special = 1;
            changed_field = 0;
            switch (specials[i])
            {
            case TC_SPECIAL_CLEAR_LINE:
                if (num_holes) 
                {
                    use_special_failed = UseSpecial(GetOwnPlayerNumber(bot_gamedata));
                    try_another_special = 1;
                    changed_field = 1;
                }
                break;

            case TC_SPECIAL_BLOCK_GRAVITY:
                if (field_height>15)
                {
                    use_special_failed = UseSpecial(GetOwnPlayerNumber(bot_gamedata));
                    try_another_special = 0;
                    changed_field = 1;
                }
                break;

            case TC_SPECIAL_NUKE:
                if (field_height>15) 
                {
                    use_special_failed = UseSpecial(GetOwnPlayerNumber(bot_gamedata));
                    try_another_special = 0;
                    changed_field = 1;
                }
                break;

            case TC_SPECIAL_SWITCH_FIELDS:
                if (field_height>15)
                {
                    if (best_field_value-field_value>100)
                    {
                        use_special_failed = UseSpecial(GetPlayerNumber(GetPlayer(bot_gamedata,best_field_index)));
                        sleep(1);
                        /* We sleep to ensure we don't do anything too 
                             fast after switching */
                    }
                    else
                    {
                        use_special_failed = UseSpecial(GetOwnPlayerNumber(bot_gamedata));
                        try_another_special = 1;
                    }
                    changed_field = 1;
                }
                break;

            case TC_SPECIAL_CLEAR_SPECIALS:
            {
                int j;
                for (j=0; j<num_players; j++)
                {
                    if (fields[j] && HasGoodSpecial(field_specials[j]))
                    {
                        use_special_failed = UseSpecial(GetPlayerNumber(GetPlayer(bot_gamedata, j)));
                        break;
                    }
                }
            }
            break;

            case TC_SPECIAL_ADD_LINE:
                use_special_failed = UseSpecial(GetPlayerNumber(GetPlayer(bot_gamedata, best_field_index)));
                try_another_special = 1;
                break;

            case TC_SPECIAL_BLOCK_BOMB:
                for (i=num_players-1; i>=0; i--)
                {
                    if (fields[i] && 
                            field_specials[i][TC_SPECIAL_BLOCK_BOMB-TC_SPECIAL_FIRST_SPECIAL])
                    {
                        use_special_failed = UseSpecial(GetPlayerNumber(GetPlayer(bot_gamedata, i)));
                    }
                    else if (field_height>15)
                    {
                        DiscardSpecial();
                        try_another_special = 1;
                    }
                }
                break;

            case TC_SPECIAL_CLEAR_RANDOM:
                use_special_failed = UseSpecial(GetPlayerNumber(GetPlayer(bot_gamedata, best_field_index)));
                try_another_special = 1;
                break;

            case TC_SPECIAL_BLOCK_QUAKE:
                use_special_failed = UseSpecial(GetPlayerNumber(GetPlayer(bot_gamedata, best_field_index)));
                try_another_special = 1;
                break;

            }
            /* Have this test here as a failsafe to make sure we exit
                 if the game is over */
            {
                int dummy1, dummy2, dummy3;
                if (GetCurrentBlock(&dummy1, &dummy2, &dummy3)) break;
            }
            if (!try_another_special || use_special_failed) break;
        }
    }
}
