/*
   Copyright (C) 2003 Christopher Yeoh <cyeoh@samba.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#define _GNU_SOURCE
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <getopt.h>
#include <libgen.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <termios.h>
#include <string.h>

#define DEFAULT_DEVICE "/dev/ttyS0"

#define COMMAND_RELAY_ON 0x01
#define COMMAND_RELAY_OFF 0x02
#define COMMAND_RELAY_TOGGLE 0x04
#define COMMAND_RELAY_OUTPUT_STATUS 0x08
#define COMMAND_RELAY_INPUT_STATUS 0x0A

char *DeviceName = NULL;


int create_lock_file()
{
	int fd;
	char *lockfile;

	asprintf(&lockfile, "/var/lock/LCK..%s", basename(DeviceName));

	if ( (fd = open(lockfile, O_RDWR|O_CREAT|O_EXCL)) < 0 )
	{
		if (errno==EEXIST)
		{
			printf("Lock file %s exists\n", lockfile);
		}
		else
		{
			perror("Could not create lock file");
		}
		return 0;
	}

	close(fd);
	return 1;
}

void remove_lock_file()
{
	char *lockfile;

	asprintf(&lockfile, "/var/lock/LCK..%s", basename(DeviceName));

	unlink(lockfile);

}

void write_string(int fd, char *string)
{
	int i;
	for (i=0; i<strlen(string); i++)
	{
		if (write(fd, &(string[i]), 1)!=1)
		{
			perror("write failed");
		}
	}
}

int get_result(int fd, int *result)
{
	int count;
	char string_result[20];
	if ( (count = read(fd, &string_result, 10)) )
	{
		char *start;
		start = strstr(string_result, "\r\n");
		start += 2;
		if ( (start[0]>47 && start[0]<58) )
		{
			*result = start[0]-48;
		}
		else if (start[0]>64 && start[0]<71)
		{
			*result = start[0]-65+10;
		}
		else
		{
			return 0;
		}
		if (start[1]>47 && start[1]<58)
		{
			*result <<= 4;
			*result |= start[1]-48;
		}
		else if (start[1]>64 && start[1]<71)
		{
			*result <<= 4;
			*result |= start[1]-65+10;
		}
		return 1;
	}
	return 0;
}

void print_result(int result, int count)
{
	int i;
	for (i=0; i<count; i++)
	{
		printf("%i: %s\n", i+1, (result>>i) & 1 ? "On" : "Off");
	}
}

int main(int argc, char *argv[])
{
	signed char option;
	int error = 0;
	int command = 0;
	int relay_output_status_check = 0;
	int relay_input_status_check = 0;
	int toggle_relay_num = 0;
	int turn_relay_on = 0;
	int turn_relay_off = 0;

	while ( (option=getopt(argc, argv, "vd:n::f::t::s::i::")) > 0)
	{
		switch (option)
		{
		case 'v':
			printf("Version 1.0\n");
			break;

		case 'd':
			asprintf(&DeviceName, "%s", optarg);
			break;

		case 's':
			command |= COMMAND_RELAY_OUTPUT_STATUS;
			if (optarg) relay_output_status_check = atoi(optarg);
			break;

		case 'i':
			command |= COMMAND_RELAY_INPUT_STATUS;
			if (optarg) relay_input_status_check = atoi(optarg);
			break;

		case 't':
			command |= COMMAND_RELAY_TOGGLE;
			if (optarg) toggle_relay_num = atoi(optarg);
			break;

		case 'n':
			command |= COMMAND_RELAY_ON;
			if (optarg) turn_relay_on = atoi(optarg);
			break;

		case 'f':
			command |= COMMAND_RELAY_OFF;
			if (optarg) turn_relay_off = atoi(optarg);
			break;

		case ';':
		case '?':
			error = 1;
			break;

		default:
			printf("Unknown option %c\n", option);
		}
	}

	if (error)
	{
		exit(1);
	}

	if (DeviceName==NULL)
	{
		asprintf(&DeviceName, DEFAULT_DEVICE);
	}
	
	/* Create lock file */
	if (create_lock_file())
	{
		int device_fd;
		char *command_string;

		/* Open device file */
		if ( (device_fd = open(DeviceName, O_RDWR | O_NOCTTY)) < 0 )
		{
			perror("Could not open device");
		}
		else
		{
			struct termios terminal;
			terminal.c_cflag = B9600 | CRTSCTS | CS8 | CREAD | CLOCAL | HUPCL;
			terminal.c_iflag = IGNBRK;
			terminal.c_oflag = 0;
			terminal.c_lflag = 0;
			terminal.c_line = 0;
			terminal.c_cc[VMIN] = 1;
			terminal.c_cc[VTIME] = 5;
 			cfsetispeed(&terminal, B9600);
			tcflush(device_fd, TCIFLUSH);
			tcsetattr(device_fd, TCSANOW, &terminal);
			
			if (command & COMMAND_RELAY_OUTPUT_STATUS)
			{
				int result;

 				asprintf(&command_string, "S%i\r", relay_output_status_check); 
				write_string(device_fd, command_string);
				if (get_result(device_fd, &result))
				{
					if (relay_output_status_check)
						printf("%s\n", result ? "On" : "Off");
					else
						print_result(result, 8);
				}
				else
				{
					printf("Failed read\n");
				}
			}
			else if (command & COMMAND_RELAY_TOGGLE)
			{
				int result;
 				asprintf(&command_string, "T%i\r", toggle_relay_num); 
				write_string(device_fd, command_string);
				get_result(device_fd, &result);
			}
			else if (command & COMMAND_RELAY_ON)
			{
				int result;
 				asprintf(&command_string, "N%i\r", turn_relay_on); 
				write_string(device_fd, command_string);
				get_result(device_fd, &result);
			}
			else if (command & COMMAND_RELAY_OFF)
			{
				int result;
 				asprintf(&command_string, "F%i\r", turn_relay_off); 
				write_string(device_fd, command_string);
				get_result(device_fd, &result);
			}
			else if (command & COMMAND_RELAY_INPUT_STATUS)
			{
				int result;

 				asprintf(&command_string, "I%i\r", relay_input_status_check); 
				write_string(device_fd, command_string);
				if (get_result(device_fd, &result))
				{
					if (relay_input_status_check)
						printf("%s\n", result ? "On" : "Off");
					else
						print_result(result, 4);
				}
				else
				{
					printf("Failed read\n");
				}
			}

			close(device_fd);
		}


		remove_lock_file();

	}
	exit(0);
}
